
/*--------------------------------------------------------------------*/
/*--- PowerPC-specific libpthread code.           ppc/libpthread.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

/* ALL THIS CODE RUNS ON THE SIMULATED CPU.

   See the comments at the top of coregrind/vg_libpthread.c for some
   caveats.
*/

#include "core.h"

#define __USE_UNIX98
#include <pthread.h>
#undef __USE_UNIX98

#define __USE_GNU
#include <dlfcn.h>
#undef __USE_GNU

#include <errno.h>

// Struct used to describe a TCB header, copied from glibc.
typedef
   struct {
      void *tcb;
      void *dtv;
      void *self;
      int multiple_threads;
   }
   tcbhead_t;

/* R2 is reserved for the thread pointer on ppc32 */
register tcbhead_t *__thread_register asm ("r2");

/* --------------------------------------------------- 
   Helper functions for running a thread 
   and for clearing up afterwards.
   ------------------------------------------------ */

typedef void *(*allocate_tls_t) (void *result);
typedef void (*deallocate_tls_t) (void *tcb, int dealloc_tcb);

static allocate_tls_t allocate_tls = NULL;
static deallocate_tls_t deallocate_tls = NULL;

Bool VGA_(has_tls)(void)
{
    return __thread_register->self != NULL;
}

void VGA_(thread_create)(arch_thread_aux_t *aux)
{
    if (VGA_(has_tls)()) {
	if (allocate_tls == NULL || deallocate_tls == NULL) {
	    allocate_tls = (allocate_tls_t)
		dlsym(RTLD_DEFAULT, "_dl_allocate_tls");
	    deallocate_tls = (deallocate_tls_t)
		dlsym(RTLD_DEFAULT, "_dl_deallocate_tls");
	}
	my_assert(allocate_tls != NULL);

	aux->tls_data = allocate_tls(NULL);

	__thread_register->multiple_threads = 1;
    } else {
	aux->tls_data = NULL;
    }
}

void VGA_(thread_wrapper)(arch_thread_aux_t *aux)
{
    if (aux->tls_data) {
	__thread_register = aux->tls_data;
	__thread_register->self = __thread_register;
	__thread_register->multiple_threads = 1;
    }
}

void VGA_(thread_exit)(void)
{
    /* Free up any TLS data */
    if (VGA_(has_tls)() && pthread_self() > 1) {
	my_assert(deallocate_tls != NULL);
	deallocate_tls(__thread_register, 1);
    }
}

typedef volatile int pthread_spinlock_t; /* Huh?  Guarded by __USE_XOPEN2K */

int pthread_spin_init(pthread_spinlock_t *lock, int pshared)
{
    *lock = 0;
    return 0;
}

int pthread_spin_lock(pthread_spinlock_t *lock)
{
    int tmp;

    asm volatile
	("\n"
	 "1:	lwarx %0,0,%1\n"
	 "	cmpwi 0,%0,0\n"
	 "	bne- 2f\n"
	 "	stwcx. %2,0,%1\n"
	 "	bne- 2f\n"
	 "	isync\n"
	 "	.subsection 1\n"
	 "2:	lwz %0,0(%1)\n"
	 "	cmpwi 0,%0,0\n"
	 "	bne 2b\n"
	 "	b 1b\n"
	 "	.previous"
	 : "=&r" (tmp) : "r" (lock), "r" (1)
	 : "cr0", "memory");
    return 0;
}

int pthread_spin_unlock(pthread_spinlock_t *lock)
{
    *lock = 0;
    return 0;
}

int pthread_spin_destroy(pthread_spinlock_t *lock)
{
  /* Nothing to do.  */
  return 0;
}

int pthread_spin_trylock(pthread_spinlock_t *lock)
{
    int tmp;

    asm volatile
	("\n"
	 "1:	lwarx %0,0,%1\n"
	 "	cmpwi 0,%0,0\n"
	 "	bne 2f\n"
	 "	stwcx. %2,0,%1\n"
	 "	bne- 1b\n"
	 "	isync\n"
	 "2:"
	 : "=&r" (tmp)
	 : "r" (lock), "r" (1)
	 : "cr0", "memory");
    return tmp? EBUSY: 0;
}
