// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_STATIC_POOL_H
#define POOMA_UTILITIES_STATIC_POOL_H

//-----------------------------------------------------------------------------
// Classes:
// StaticPool
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Overview:
//
// StaticPool
//       A class that manages a static Pool in which the block size is
//       a template parameter.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "Utilities/Pool.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
// If you just create a Pool as a static object in each of many different
// pooled classes, you end up with potentially a large number of different 
// pools. In particular, if you pool of expression objects, you will have a
// different pool for each kind of expression object, which is inefficient
// because many different expression types will have the same size, and
// could therefore share a pool.
//
// class StaticPool<S> has a static Pool of size S.  Strictly speaking,
// it has a pool of size S', where S' is rounded up to a multiple of 8 bytes.
// All the StaticPools that round up to size S' share the same pool.
//
// This is done by having the Pool be static data in a base class 
// RoundedStaticPool<SP>, where SP is S rounded up.
//
// Usage: When you want a chunk of memory for an object of type T, you say:
//
//     T* p = StaticPool<sizeof(T)>::alloc() 
//
// To free that memory you say:
//
//     StaticPool<sizeof(T)>::free(p);
//
//-----------------------------------------------------------------------------

//
// class RoundedStaticPool
// All this does is define alloc and free as static functions,
// and the static pool itself.
//
//

template<int SP>
class RoundedStaticPool
{
public:

  // Get a block of memory.
  static void *alloc() { return pool_s.alloc(); }

  // Return a block of memory.
  static void free(void *p) { pool_s.free(p); }

private:

  // This class stores the pool.
  static Pool pool_s;

  // Forbid construction by making this private.
  RoundedStaticPool() {}

};

//
// Declare the storage for the static pool.
//

template<int SP> Pool RoundedStaticPool<SP>::pool_s(SP);

//----------------------------------------------------------------------
//
// class StaticPool
//
// This is a wrapper class on RoundedStaticPool, which just rounds up
// its input block size and inherits from RoundedStaticPool.
//
// It doesn't need to do anything else since it inherits the alloc
// and free functions.
//
//----------------------------------------------------------------------

template<class T>
class StaticPool 
: public RoundedStaticPool<(sizeof(T)%8 ? sizeof(T)+8-(sizeof(T)%8) : sizeof(T)) >
{
private:

  // Forbid construction by making this private:
  StaticPool() {}

};

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: StaticPool.h,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/03/07 13:18:28 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
