// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_DEREFITERATOR_H
#define POOMA_UTILITIES_DEREFITERATOR_H

//-----------------------------------------------------------------------------
// Classes: 
//   DerefIterator
//   ConstDerefIterator
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Overview:
//   DerefIterator<T>
//   ConstDerefIterator<T>
//     - STL style iterators for lists of pointers. Unlike vector<T*>::iterator,
//       these automatically dereference themselves and, in the process, 
//       maintain const correctness.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include <vector>


///////////////////////////////////////////////////////////////////////////////
// namespace Pooma {

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

template <class T>
class ConstDerefIterator;

//-----------------------------------------------------------------------------
// Full Description:
// DerefIterator<T>
//
// DerefIterator<T> and ConstDerefIterator<T> are STL-style iterators that
// are used to properly handle iterating through lists of pointers. Not only
// is this a convenience, as these iterators automatically dereference
// themselves, it also solves a problem with const correctness. If one has
// vector<T*>::const_iterator, this only keeps the user from modifying the
// pointer, not from modifying the object that is pointed to. What one really
// wants is vector<const T*>::const_iterator, but that is not something
// one can get from vector<T*>. 
//-----------------------------------------------------------------------------

template <class T>
class DerefIterator
{
public:

  // Convenience typedefs
  
  typedef T                                   Value_t;
  typedef std::vector<T*>                     List_t;
  typedef DerefIterator<Value_t>              iterator;
  typedef ConstDerefIterator<Value_t>         const_iterator;
  
  // Required iterator typedefs
  
  typedef std::random_access_iterator_tag     iterator_category;
  typedef Value_t                             value_type;
  typedef ptrdiff_t                           difference_type;
  typedef Value_t*                            pointer;
  typedef Value_t&                            reference;

  friend class ConstDerefIterator<T>;

  friend iterator operator+ (const difference_type n, const iterator &iter)
  {
    return iterator(iter.p_m + n);
  }

protected:

  typename List_t::iterator p_m;

public:

  inline DerefIterator() { }
  inline DerefIterator(typename List_t::iterator x) : p_m(x) { }

  inline reference operator*() const {
    return **p_m; 
  }
  inline pointer operator->() const {
    return *p_m; 
  }
  inline iterator &operator++() {
    ++p_m;
    return *this;
  }
  inline iterator operator++(int) {
    iterator tmp = *this;
    ++p_m;
    return tmp;
  }
  inline iterator &operator--() {
    --p_m;
    return *this;
  }
  inline iterator operator--(int) {
    iterator tmp = *this;
    --p_m;
    return tmp;
  }
  inline iterator &operator+=(const difference_type i) {
    p_m += i;
    return *this;
  }
  inline iterator &operator-=(const difference_type i) {
    p_m -= i;
    return *this;
  }
  inline iterator operator+(const difference_type i) const {
    return iterator(p_m + i);
  }
  inline iterator operator-(const difference_type i) const {
    return iterator(p_m - i);
  }
  inline difference_type operator-(const iterator &x) const {
    return p_m - x.p_m;
  }
  inline difference_type operator-(const const_iterator &x) const {
    return p_m - x.p_m;
  }
  inline reference operator[](const difference_type i) const {
    return **(p_m + i); 
  }

  inline bool operator==(const iterator &x) const {
    return p_m == x.p_m;
  }
  inline bool operator==(const const_iterator &x) const {
    return p_m == x.p_m;
  }
  inline bool operator<(const iterator &x) const {
    return p_m < x.p_m;
  }
  inline bool operator<(const const_iterator &x) const {
    return p_m < x.p_m;
  }
  inline bool operator!= (const iterator &y) const
  { return ! (*this == y); }
  inline bool operator>  (const iterator &y) const
  { return (y < *this); }
  inline bool operator<= (const iterator &y) const
  { return  ! (y < *this); }
  inline bool operator>= (const iterator &y) const
  { return  ! (*this < y); }
  inline bool operator!= (const const_iterator &y) const
  { return  ! (*this == y); }
  inline bool operator>  (const const_iterator &y) const
  { return  (y < *this); }
  inline bool operator<= (const const_iterator &y) const
  { return  ! (y < *this); }
  inline bool operator>= (const const_iterator &y) const
  { return  ! (*this < y); }
};

template <class T>
class ConstDerefIterator
{
public:

  // Convience typedefs
  
  typedef T                                   Value_t;
  typedef std::vector<T*>                     List_t;
  typedef DerefIterator<Value_t>              iterator;
  typedef ConstDerefIterator<Value_t>         const_iterator;
  
  // Required typedefs
  
  typedef std::random_access_iterator_tag     iterator_category;
  typedef Value_t                             value_type;
  typedef ptrdiff_t                           difference_type;
  typedef const Value_t*                      pointer;
  typedef const Value_t&                      reference;

  friend class DerefIterator<T>;

  friend const_iterator operator+ (const difference_type n, 
    const const_iterator &iter) 
  {
    return const_iterator(iter.p_m + n);
  }

protected:

  typename List_t::const_iterator p_m;

public:

  inline ConstDerefIterator() { }
  inline ConstDerefIterator(const typename List_t::const_iterator &x) : p_m(x) { }
  inline ConstDerefIterator(const typename List_t::iterator &x) : p_m(x) { }
  inline ConstDerefIterator(const iterator &x) : p_m(x.p_m) { }
  inline ConstDerefIterator(const const_iterator &x) : p_m(x.p_m) { }

  inline reference operator*() const {
    return **p_m; 
  }
  inline pointer operator->() const {
    return *p_m; 
  }
  inline const_iterator &operator++() {
    ++p_m;
    return *this;
  }
  inline const_iterator operator++(int) {
    const_iterator tmp = *this;
    ++p_m;
    return tmp;
  }
  inline const_iterator &operator--() {
    --p_m;
    return *this;
  }
  inline const_iterator operator--(int) {
    const_iterator tmp = *this;
    --p_m;
    return tmp;
  }
  inline const_iterator &operator+=(const difference_type i) {
    p_m += i;
    return *this;
  }
  inline const_iterator &operator-=(const difference_type i) {
    p_m -= i;
    return *this;
  }
  inline const_iterator operator+(const difference_type i) const {
    return const_iterator(p_m + i);
  }
  inline const_iterator operator-(const difference_type i) const {
    return const_iterator(p_m - i);
  }
  inline difference_type operator-(const const_iterator &x) const {
    return p_m - x.p_m;
  }
  inline difference_type operator-(const iterator &x)
    const {
    return p_m - x.p_m;
  }
  inline reference operator[](const difference_type i) const {
    return **(p_m + i); 
  }

  inline bool operator==(const const_iterator &x) const {
    return p_m == x.p_m;
  }
  inline bool operator==(const iterator &x) const {
    return p_m == x.p_m;
  }
  inline bool operator<(const const_iterator &x) const {
    return p_m < x.p_m;
  }
  inline bool operator<(const iterator &x) const {
    return p_m < x.p_m;
  }
  inline bool operator!= (const const_iterator &y) const
  { return ! (*this == y); }
  inline bool operator>  (const const_iterator &y) const
  { return (y < *this); }
  inline bool operator<= (const const_iterator &y) const
  { return  ! (y < *this); }
  inline bool operator>= (const const_iterator &y) const
  { return  ! (*this < y); }
  inline bool operator!= (const iterator &y) const
  { return  ! (*this == y); }
  inline bool operator>  (const iterator &y) const
  { return  (y < *this); }
  inline bool operator<= (const iterator &y) const
  { return  ! (y < *this); }
  inline bool operator>= (const iterator &y) const
  { return  ! (*this < y); }
};

// } // namespace POOMA

#endif // POOMA_UTILITIES_DEREFITERATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DerefIterator.h,v $   $Author: swhaney $
// $Revision: 1.4 $   $Date: 2000/03/07 13:18:23 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
