// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// RemoteProxy<T>
//-----------------------------------------------------------------------------

#ifndef POOMA_CHEETAH_REMOTE_PROXY_H
#define POOMA_CHEETAH_REMOTE_PROXY_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Tulip/Messaging.h"
#include "Functions/ComponentAccess.h"

#if POOMA_CHEETAH
# include "Cheetah/Cheetah.h"
#endif

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template <int Dim> class Loc;

//-----------------------------------------------------------------------------
//
// RemoteProxy<T>
//
// This class is the return type of the remote brick engine operator().
// We need an object that lets us assign to data on this context, but that
// can also contain the data that came from another context, and that prevents
// you from writing to that data.
//
// Outstanding:
//
// 1) maybe need a copy constructor etc.  right now it's ok as long as proxies
// don't live too long.
//
// 2) other operator functions.
//
// Usage:
//
// A RemoteProxy must be constructed with a value and the context that the
// value belongs to.
//
//-----------------------------------------------------------------------------

struct RemoteProxyBase
{
  // If we need a remote value, then this flag lets us know when it's
  // ready.  This value is static because it is used to block the parse
  // thread until the data is received.

  static bool ready_m;

  // We only need one tag for all the remote proxies.  Perhaps this could
  // be packaged with the handler for remote proxies.

  static int tag_m;
};

template<class T>
class RemoteProxy
{
public:
  typedef RemoteProxy<T> This_t;

  // All the work happens in the remote proxy constructor.
  // If we're on the right context, we store a pointer to the
  // value and broadcast the value to the other contexts.
  // Otherwise we receive the value from the owning context.

  RemoteProxy(T &val, int owningContext = 0)
  {
    int tag = RemoteProxyBase::tag_m++;
    if (Pooma::context() == owningContext)
    {
      value_m = &val;

#if POOMA_CHEETAH
      int toContext;
      for (toContext = 0; toContext < Pooma::contexts(); ++toContext)
      {
	if (toContext != Pooma::context())
	{
	  Pooma::indexHandler()->send(toContext, tag, val);
	}
      }
#endif
    }
    else
    {
#if POOMA_CHEETAH
      storedValue_m = val;
      value_m = &storedValue_m;

      RemoteProxyBase::ready_m = false;

      Pooma::indexHandler()->request(owningContext, tag, receive, this);

      while (!RemoteProxyBase::ready_m)
      {
	Pooma::poll();
      }
#endif
    }
  }

  RemoteProxy(const RemoteProxy<T> &s)
  {
    if (s.value_m != &s.storedValue_m)
    {
      value_m = s.value_m;
    }
    else
    {
      storedValue_m = s.value();
      value_m = &storedValue_m;
    }
  }

  inline
  operator T() const { return *value_m; }

  inline
  T &value() { return *value_m; }

  inline
  const T &value() const { return *value_m; }

  template<class S>
  inline
  RemoteProxy<T> &operator=(const S &s)
  {
    *value_m = s;
    return *this;
  }

  template<class S>
  inline
  RemoteProxy<T> &operator=(const RemoteProxy<S> &s)
  {
    *value_m = s.value();
    return *this;
  }

  inline
  RemoteProxy<T> &operator=(const RemoteProxy<T> &s)
  {
    *value_m = s.value();
    return *this;
  }

  inline
  typename ComponentAccess<T, Loc<1> >::Element_t
  operator()(int i) const
  {
    return ComponentAccess<T, Loc<1> >::index(value(), Loc<1>(i));
  }

  inline
  typename ComponentAccess<T, Loc<1> >::ElementRef_t
  operator()(int i)
  {
    return ComponentAccess<T, Loc<1> >::indexRef(value(), Loc<1>(i));
  }

private:

  // Handler function for Cheetah.

  static void receive(This_t *me, T &value)
  {
    me->storedValue_m = value;
    RemoteProxyBase::ready_m = true;
  }
 
 // Pointer to the actual value represented by this proxy.

  T *value_m;

  // If we get a remote value, we store it here.

  T storedValue_m;

};

//-----------------------------------------------------------------------------
// Some mathematical operations.
// These probably need to be improved to promote the types.
//-----------------------------------------------------------------------------

template<class T, class S>
inline T
operator*(const RemoteProxy<T> &t, const S &s)
{
  return t.value() * s;
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_CHEETAH_REMOTE_PROXY_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RemoteProxy.h,v $   $Author: sa_smith $
// $Revision: 1.13 $   $Date: 2000/06/08 22:16:59 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
