// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Various tests of transpose(Tensor<>) global function (transposeerminant).
// ----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

// Forward declarations:
template<int Dim>
void testTranspose(Pooma::Tester &tester);


//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  testTranspose<1>(tester);
  testTranspose<2>(tester);
  testTranspose<3>(tester);

  int ret = tester.results("TestTranspose");
  Pooma::finalize();
  return ret;
}

template<int Dim>
void testTranspose(Pooma::Tester &tester)
{
  // Create the physical Domains:
  const int nVerts = 6;
  const int nCells = nVerts - 1;
  int nCellsTot = 1;
  Interval<Dim> vertexDomain;
  for (int d = 0; d < Dim; d++) {
    vertexDomain[d] = Interval<1>(nVerts);
    nCellsTot *= nCells;
  }

  // Create the (uniform, logically rectilinear) mesh.
  Vector<Dim> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);

  // Create geometry object:
  typedef DiscreteGeometry<Cell, Mesh_t> Geometry_t;
  Geometry_t geom(mesh);

  // Create the Fields:

  // Full, Antisymmetric, Symmetric, Diagonal Tensor Fields:
  Field<Geometry_t,Tensor<Dim,double,Full> > tff(geom);
  Field<Geometry_t,Tensor<Dim,double,Symmetric> > tfs(geom);
  Field<Geometry_t,Tensor<Dim,double,Antisymmetric> > tfa(geom);
  Field<Geometry_t,Tensor<Dim,double,Diagonal> > tfd(geom);

  // Assign values:
  Tensor<Dim,double,Full> tf(0.0), tfTranspose;
  Tensor<Dim,double,Symmetric> ts(0.0), tsTranspose;
  Tensor<Dim,double,Antisymmetric> ta(0.0), taTranspose;
  Tensor<Dim,double,Diagonal> td(0.0), tdTranspose;
  for (int i = 0; i < Dim; i++) {
    for (int j = 0; j < Dim; j++) {
      tf(i,j) = (i+1)*(i+1) + (j+1)*(j+1) + (i+4)*(j+4) + i;
      tfTranspose(j,i) = tf(i,j);
    }
  }
  ts = symmetrize<Symmetric>(tf);
  ta = symmetrize<Antisymmetric>(tf);
  td = symmetrize<Diagonal>(tf);
  tff = tf;
  tfs = ts;
  tfa = ta;
  tfd = td;
  tsTranspose = ts;
  tdTranspose = td;
  for (int i = 1; i < Dim; i++) {
    for (int j = 0; j < i;  j++) {
      taTranspose(i,j) = -ta(i,j);
    }
  }
  tester.out() << "tf = " << tf << std::endl;
  tester.out() << "ts = " << ts << std::endl;
  tester.out() << "ta = " << ta << std::endl;
  tester.out() << "td = " << td << std::endl;

  // Test transpose of Full Tensor:
  Tensor<Dim,double,Full> transposeValF;
  transposeValF = sum(transpose(tff));
  if (!tester.check("transposeValF", transposeValF, tfTranspose*nCellsTot)) {
    tester.out() << Dim << "D, sum(transpose(tff)) = " << transposeValF
                 << " != tfTransposenCellsTot = " 
                 << tfTranspose*nCellsTot << std::endl;
  }

  // Test transpose of Symmetric Tensor:
  Tensor<Dim,double,Symmetric> transposeValS;
  transposeValS = sum(transpose(tfs));
  if (!tester.check("transposeValS", transposeValS, tsTranspose*nCellsTot)) {
    tester.out() << Dim << "D, sum(transpose(tfs)) = " << transposeValS
                 << " != tsTranspose*nCellsTot = " 
                 << tsTranspose*nCellsTot << std::endl;
  }

  // Test transpose of Antiymmetric Tensor:
  Tensor<Dim,double,Antisymmetric> transposeValA;
  transposeValA = sum(transpose(tfa));
  if (!tester.check("transposeValA", transposeValA, taTranspose*nCellsTot)) {
    tester.out() << Dim << "D, sum(transpose(tfa)) = " << transposeValA
                 << " != taTranspose*nCellsTot = " 
                 << taTranspose*nCellsTot << std::endl;
  }

  // Test transpose of Diagonal Tensor:
  Tensor<Dim,double,Diagonal> transposeValD;
  transposeValD = sum(transpose(tfd));
  if (!tester.check("transposeValD", transposeValD, tdTranspose*nCellsTot)) {
    tester.out() << Dim << "D, sum(transpose(tfd)) = " << transposeValD
                 << " != tdTranspose*nCellsTot = " 
                 << tdTranspose*nCellsTot << std::endl;
  }

}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestTranspose.cpp,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/03/07 13:18:19 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
