// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_TINY_ZERO_H
#define POOMA_TINY_ZERO_H

//-----------------------------------------------------------------------------

// Class: Zero
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
// A numeric class for a number that is always zero.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
// The point of this class is to be a number of type T which is known at
// compile time to be zero.  This is reflected in the operations like
// addition and subtraction that use it.  If you return a Zero object,
// the compiler should be able to make better optimizations than if you
// just return zero.
//
// Zero is templated on type T, to represent a zero object of type T.
// Type T is required to be constructable from an integer zero.
//
//-----------------------------------------------------------------------------

template<class T>
class Zero
{
  // If you need to convert to an object of type T,
  // just build one from zero.  This will be used in 
  // the cases where the operators below don't match.

  operator T() { return T(0); }


  // Give it empty ctors and assignment operators
  // to try and keep purify happy.

  Zero() {}
  Zero(const Zero<T>&) {}
  Zero<T>& operator=(const Zero<T>&) { return *this; }
};

//-----------------------------------------------------------------------------
//
// Forward declarations for classes used by TypeComputations for figuring out
// return types in expression templates.
//
//-----------------------------------------------------------------------------

template<class T, class Op> struct UnaryReturn;
template<class T1, class T2, class Op> struct BinaryReturn;

//-----------------------------------------------------------------------------
//
// Operators using Zero.
//
//-----------------------------------------------------------------------------

//
// Binary multiply by a Zero<T> and T returns Zero<T>
//

template<class T> 
  inline Zero<T>  operator*(Zero<T>, const T&)   { return Zero<T>(); }

template<class T> 
  inline Zero<T>  operator*(const T&, Zero<T>)   { return Zero<T>(); }

template<class T> 
  inline Zero<T>  operator*(Zero<T>, Zero<T>)    { return Zero<T>(); }

template<class T> 
  inline Zero<T>  operator/(Zero<T>, const T&)   { return Zero<T>(); }

//
// Trait classes so that expression templates will deal correctly
// with Zeros in multiplicative operations.
//

template<class T>
struct BinaryReturn< Zero<T> , T , OpMultiply >
{
  typedef Zero<T> Type_t;
};

template<class T>
struct BinaryReturn< T, Zero<T> , OpMultiply >
{
  typedef Zero<T> Type_t;
};

template<class T>
struct BinaryReturn< Zero<T>, Zero<T> , OpMultiply >
{
  typedef Zero<T> Type_t;
};

template<class T>
struct BinaryReturn< Zero<T> , T , OpDivide >
{
  typedef Zero<T> Type_t;
};

//
// Adding a Zero<T> to a T returns the T.
//

template<class T> 
  inline const T& operator+(Zero<T>, const T& x) { return  x; }

template<class T> 
  inline const T& operator+(const T& x, Zero<T>) { return  x; }

template<class T> 
  inline Zero<T>  operator+(Zero<T>, Zero<T>)    { return  Zero<T>(); }

template<class T> 
  inline T        operator-(Zero<T>, const T& x) { return -x; }

template<class T> 
  inline const T& operator-(const T& x, Zero<T>) { return  x; }

template<class T> 
  inline Zero<T>  operator-(Zero<T>, Zero<T>)    { return  Zero<T>(); }

//
// Trait class specializations for additive operations.
//

template<class T>
struct BinaryReturn< Zero<T> , T , OpAdd >
{
  typedef T Type_t;
};

template<class T>
struct BinaryReturn< T, Zero<T> , OpAdd >
{
  typedef T Type_t;
};

template<class T>
struct BinaryReturn< Zero<T>, Zero<T> , OpAdd >
{
  typedef Zero<T> Type_t;
};

template<class T>
struct BinaryReturn< Zero<T> , T , OpSubtract >
{
  typedef T Type_t;
};

template<class T>
struct BinaryReturn< T, Zero<T> , OpSubtract >
{
  typedef T Type_t;
};

template<class T>
struct BinaryReturn< Zero<T>, Zero<T> , OpSubtract >
{
  typedef Zero<T> Type_t;
};

//
// Unary Minus of a zero returns a zero.
//

template<class T>
  inline Zero<T>  operator-(Zero<T>) { return Zero<T>(); }

template<class T>
struct UnaryReturn< Zero<T> , OpUnaryMinus >
{
  typedef Zero<T> Type_t;
};


//
// Unary Plus of a zero returns a zero.
//

template<class T>
  inline Zero<T>  operator+(Zero<T>) { return Zero<T>(); }

template<class T>
struct UnaryReturn< Zero<T> , OpUnaryPlus >
{
  typedef Zero<T> Type_t;
};

//
// Trait class definitios for the cases where the Zero<T> gets
// converted to a T.
//

template<class T, class Op>
struct UnaryReturn< Zero<T> , Op >
{
  typedef typename UnaryReturn<T,Op>::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct BinaryReturn< Zero<T1> , T2, Op >
{
  typedef typename BinaryReturn<T1,T2,Op>::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct BinaryReturn< T1 , Zero<T2>, Op >
{
  typedef typename BinaryReturn<T1,T2,Op>::Type_t Type_t;
};

template<class T1, class T2, class Op>
struct BinaryReturn< Zero<T1> , Zero<T2>, Op >
{
  typedef typename BinaryReturn<T1,T2,Op>::Type_t Type_t;
};



#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Zero.h,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/03/07 13:18:17 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
