// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_TINY_VECTOR_ELEMENTS_H
#define POOMA_TINY_VECTOR_ELEMENTS_H

//-----------------------------------------------------------------------------

// Class: 
//    VectorElem
//    VectorAssign
//    VectorBinaryCombine
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
// Trait classes for getting elements of Tiny objects at compile time.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<int D, class T, class E> class Vector;
template<int D, class T, class E> class VectorEngine;

//-----------------------------------------------------------------------------
//
// Full Description:
//
// The general templates for the class VectorElem.
// VectorElem should be specialized for vector-like classes.
//
// The general definition is for scalars which cannot be subscripted.
//
// The reason for this is so that you can have an object that acts like
// an n-dimensional vector but has some components generated algorithmicly.
// For example, it might have only one nonzero component.  In that case
// VectorElem would be specialized for that class and would return a Zero
// for all but the one nonzero component.
//
//-----------------------------------------------------------------------------

template<class V, int I>
struct VectorElem
{
  typedef V Element_t;
  typedef const V& ConstElementRef_t;
  typedef V& ElementRef_t;
  static const V& get(const V& x) { return x; }
  static       V& get(      V& x) { return x; }
};

template<int D, class T, class E, int I>
struct VectorEngineElem
{
  typedef VectorEngine<D,T,E> V;
  typedef typename V::Element_t Element_t;
  typedef typename V::ConstElementRef_t ConstElementRef_t;
  typedef typename V::ElementRef_t ElementRef_t;
  static ConstElementRef_t get(const V& x) { return x(I); }
  static ElementRef_t      get(      V& x) { return x(I); }
};

template<int D, class T, class E, int I>
struct VectorElem< Vector<D,T,E> , I >
{
  typedef Vector<D,T,E> V;
  typedef VectorEngineElem<D,T,E,I> VE;
  typedef typename VE::Element_t Element_t;
  typedef typename VE::ConstElementRef_t ConstElementRef_t;
  typedef typename VE::ElementRef_t ElementRef_t;
  static ConstElementRef_t get(const V& x) { return VE::get(x.engine()); }
  static ElementRef_t get(V& x) { return VE::get(x.engine()); }
};

//-----------------------------------------------------------------------------
//
// VectorAssign
//
// Template metaprogram for copying out of one vector and into another.
// Input:
//   The vector we're writing into.
//   Something to copy out of.
//
// Evaluate by recursing on the first half and the second half.
// Terminate at either length 1 or length 2.
//
//-----------------------------------------------------------------------------

template<class V1, class V2, class Op, int B, int L>
struct VectorAssign
{
  static void apply(V1& v1, const V2& v2, Op op)
    {
      CTAssert(L>1);
      VectorAssign<V1,V2,Op,B,L/2>::apply(v1,v2,op);
      VectorAssign<V1,V2,Op,B+L/2,L-L/2>::apply(v1,v2,op);
    }
};

template<class V1, class V2, class Op, int B>
struct VectorAssign<V1,V2,Op,B,0>
{
  static void apply(V1&, const V2&, Op) {}
};

template<class V1, class V2, class Op, int B>
struct VectorAssign<V1,V2,Op,B,1>
{
  static void apply(V1& v1, const V2& v2, Op op) 
    {
      op(VectorElem<V1,B>::get(v1),VectorElem<V2,B>::get(v2));
    }
};

template<class V1, class V2, class Op, int B>
struct VectorAssign<V1,V2,Op,B,2>
{
  static void apply(V1& v1, const V2& v2, Op op) 
    {
      op(VectorElem<V1,B  >::get(v1), VectorElem<V2,B  >::get(v2));
      op(VectorElem<V1,B+1>::get(v1), VectorElem<V2,B+1>::get(v2));
    }
};

template<class V1, class V2, class Op, int B>
struct VectorAssign<V1,V2,Op,B,3>
{
  static void apply(V1& v1, const V2& v2, Op op) 
    {
      op(VectorElem<V1,B  >::get(v1), VectorElem<V2,B  >::get(v2));
      op(VectorElem<V1,B+1>::get(v1), VectorElem<V2,B+1>::get(v2));
      op(VectorElem<V1,B+2>::get(v1), VectorElem<V2,B+2>::get(v2));
    }
};

//-----------------------------------------------------------------------------
//
// Template metaprogram for combining two vectors into a third
// using a PETE operator tag.
//
// The template parameters are:
// V0: The type for the destination vector.
// V1: The type for the first source vector.
// V2: The type for the second source vector.
// Op: The type for the operator tag.
// D:  An integer for the number of dimensions.
//
// The operator tag must have an operator() defined which
// takes in an element from each of the two source vectors
// and returns the element to be stored in the destination.
//
// The number of elements must be >= 0.
//
// This metaprogram operates in D dimensions by recursing on the first
// and second half.
//
//-----------------------------------------------------------------------------

template<class V0, class V1, class V2, class Op, int B, int L>
struct VectorBinaryCombine
{
  static void apply(V0& v0, const V1& v1, const V2& v2, Op op)
    {
      CTAssert(L>1);
      VectorBinaryCombine<V0,V1,V2,Op,B,B+L/2>::apply(v0,v1,v2,op);
      VectorBinaryCombine<V0,V1,V2,Op,B+L/2,L-L/2>::apply(v0,v1,v2,op);
    }
};

//
// Specialization for D==0
//

template<class V0, class V1, class V2, class Op, int B>
struct VectorBinaryCombine<V0,V1,V2,Op,B,0>
{
  static void apply(V0& v0, const V1& v1, const V2& v2, Op op) {}
};

//
// Specialization for D==1
//

template<class V0, class V1, class V2, class Op, int B>
struct VectorBinaryCombine<V0,V1,V2,Op,B,1>
{
  static void apply(V0& v0, const V1& v1, const V2& v2, Op op)
    {
      VectorElem<V0,0>::get(v0) =
	op(VectorElem<V1,B>::get(v1),VectorElem<V2,B>::get(v2));
    }
};

//
// Specialization for D==2
//

template<class V0, class V1, class V2, class Op, int B>
struct VectorBinaryCombine<V0,V1,V2,Op,B,2>
{
  static void apply(V0& v0, const V1& v1, const V2& v2)
    {
      VectorElem<V0,B  >::get(v0) =
	op(VectorElem<V1,B  >::get(v1),VectorElem<V2,B  >::get(v2));
      VectorElem<V0,B+1>::get(v0) =
	op(VectorElem<V1,B+1>::get(v1),VectorElem<V2,B+1>::get(v2));
    }
};

//
// Specialization for D==3
//

template<class V0, class V1, class V2, class Op, int B>
struct VectorBinaryCombine<V0,V1,V2,Op,B,3>
{
  static void apply(V0& v0, const V1& v1, const V2& v2)
    {
      VectorElem<V0,B  >::get(v0) =
	op(VectorElem<V1,B  >::get(v1),VectorElem<V2,B  >::get(v2));
      VectorElem<V0,B+1>::get(v0) =
	op(VectorElem<V1,B+1>::get(v1),VectorElem<V2,B+1>::get(v2));
      VectorElem<V0,B+2>::get(v0) =
	op(VectorElem<V1,B+2>::get(v1),VectorElem<V2,B+2>::get(v2));
    }
};

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: VectorElements.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:18:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
