// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_THREADS_SMARTSSTUBS_H
#define POOMA_THREADS_SMARTSSTUBS_H

//-----------------------------------------------------------------------------
// Functions:
//   SimpleSerialScheduler
//   SimpleSerialScheduler::Iterate
//   SimpleSerialScheduler::DataObject
//   template<class T> class IterateScheduler
//   IterateScheduler<Stub>
//   Iterate<Stub>
//   DataObject<Stub>
//   void concurrency(int)
//   int concurrency()
//   wait
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//----------------------------------------------------------------------
// The templated classes.
// This is sort of like an abstract base class, since it doesn't 
// implement anything and you can't build one of these directly.
//----------------------------------------------------------------------

namespace Smarts {

template<class T>
class IterateScheduler
{
private:
  // Private ctor means you can't build one of these.
  IterateScheduler() {}
};

template<class T>
class Iterate
{
private:
  // Private ctor means you can't build one of these.
  Iterate() {}
};

template<class T>
class DataObject
{
private:
  // Private ctor means you can't build one of these.
  DataObject() {}
};

//----------------------------------------------------------------------
// The tag class we'll use for the template parameter.
//----------------------------------------------------------------------

class Stub
{
public:
  enum Action { Read, Write };
};

template<> class Iterate<Stub>;
template<> class IterateScheduler<Stub>;
template<> class DataObject<Stub>;

////////////////////////////////////////////////////////////////////////
//
// The specialization of Iterate for Stub
//
////////////////////////////////////////////////////////////////////////

template<>
class Iterate<Stub>
{
public:
    // Construct the Iterate with:
    //   The scheduler it will inform when it is ready.
    //   Its affinity.
  inline Iterate(IterateScheduler<Stub> & scheduler, int affinity=-1);

  // The dtor is virtual because the subclasses will need to add to it.
  virtual ~Iterate() {}
  virtual void run() = 0;
  int affinity() { return 0; }
  int hintAffinity() { return 0; }

  void affinity(int) {}
  void hintAffinity(int) {}
  void generation(int gen) { generation_m=gen; }
  int generation() { return generation_m; }
  
protected:
  IterateScheduler<Stub> &scheduler_m;

private:
  int generation_m;

};


////////////////////////////////////////////////////////////////////////////
// The specialization of IterateScheduler for Stub                        //
////////////////////////////////////////////////////////////////////////////

template<>
class IterateScheduler<Stub>
{
public:
  //---------------------------------------------------------------------------
  // Some type definition used in all Schedulers
  //---------------------------------------------------------------------------
  // Tell the scheduler that we are beginning a new generation.
  inline
  void beginGeneration() { generation_m++; }

  // Tell the scheduler that we are finishing the current generation.
  inline
  void endGeneration() {}

  inline
  void blockingEvaluate() {}

  inline
  void releaseIterates() { }

  //---------------------------------------------------------------------------
  // Constructors & Destructors;
  //---------------------------------------------------------------------------
  
  inline
  IterateScheduler() 
    : generation_m(0) 
  { }

  inline
  ~IterateScheduler() {}

  // Return the current generation.
  inline int generation() const {  return generation_m; }

  // This class simply runs an iterate as soon as it is handed off.

  inline
  void handOff(Iterate<Stub>* it)
  {
    it->run();
    delete it;
  }
  
protected:
private:
  // Record the current generation.
  int generation_m;

}; // class IterateScheduler<Stub>

inline Iterate<Stub>::Iterate(IterateScheduler<Stub> & scheduler, int)
  : scheduler_m(scheduler) 
{
  generation(scheduler.generation());
}


////////////////////////////////////////////////////////////////////////
//
// The specialization of DataObject for Stub                        //// 
// Concrete class for holding the access requests to a user object. ////
//
////////////////////////////////////////////////////////////////////////

template<>
class DataObject<Stub>
{
public:

    // There are two ways data can be used: to read or to write.
    // Don't change this to give more than two states:
    // things inside depend on that.

    // Construct the data object with an empty set of requests
    // and the given affinity.
    inline DataObject(int=-1) {}

    // Get the affinity.
    inline int affinity() const { return 0; };

    // Set the affinity.
    inline void affinity(int) {}

    // An iterate makes a request for a certain action in a certain
    // generation.
    inline void request(Iterate<Stub>&, Stub::Action) {}

    // An iterate finishes and tells the DataObject it no longer needs
    // it.  If this is the last release for the current set of requests,
    // have the IterateScheduler release some more.
    inline void release(Stub::Action) {}

protected:
private:

};

inline void concurrency(int)
{
}

inline int concurrency()
{
  return 1;
}

inline void wait()
{
}

class Runnable
{
public:
  // Runnable just takes affinity.
  inline Runnable(int affinity)
    : affinity_m(affinity)
  { }

  // The dtor is virtual because the subclasses will need to add to it.
  virtual ~Runnable() {}
  virtual void run() = 0;

  int affinity() { return affinity_m; }
  int hintAffinity() { return affinity_m; }
  void affinity(int) {}
  void hintAffinity(int) {}

private:
  int affinity_m;
};

inline void add(Runnable *runnable)
{
  runnable->run();
  delete runnable;
}

} // namespace Smarts


#endif // POOMA_THREADS_SMARTSSTUBS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: SmartsStubs.h,v $   $Author: sa_smith $
// $Revision: 1.16 $   $Date: 2000/05/31 23:26:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
