// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Classes:
// MakeFieldReturn<T>
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDMAKERETURN_H
#define POOMA_NEWFIELD_FIELDMAKERETURN_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

struct FarLeftTag;

template<class G, class T, class E> class Field;


//-----------------------------------------------------------------------------
// These are the external traits classes that are used to build trees.
// CreateLeaf is used to convert arbitrary classes into expression objects.
// MakeFieldReturn is used to combine expressions together with operators.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// MakeFieldReturn is a tool used by operator functions to construct the
// expression tree representing that function.  Each function needs to define
// a corresponding operator functor Op which is used to compute the return
// type.  The required interface for MakeFieldReturn is:
//
// typedef ... Expression_t;    // type of the expression (UnaryNode<...>)
// Expression_t make(const T&); // construct the tree
//
// These versions are a little more complicated than those for Array because we
// want to preserve Geometry information to the largest extent possible.
//-----------------------------------------------------------------------------

template<class Expr> 
struct MakeFieldReturn;

//-----------------------------------------------------------------------------
// op(Expression)
//
// This includes all combinations of scalars, arrays, and fields.
//-----------------------------------------------------------------------------

template<class Op, class Leaf>
struct MakeFieldReturn<UnaryNode<Op, Leaf> >
{
  // The node type.
  
  typedef UnaryNode<Op, Leaf> Tree_t;

  // Deduce the template parameters of the expression engine we're building.
  
  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename ForEach<Tree_t, EvalLeaf<dim>, OpCombine>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;
  typedef typename ForEach<Tree_t, FarLeftTag, FarLeftTag>::
    Type_t::GeometryTag_t GeometryTag_t;

  // Construct the type of the Field we're making.

  typedef Field<GeometryTag_t, T_t, ExpressionTag<Tree_t> > Expression_t;

  // This function turns the tree node into a Field.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};


//-----------------------------------------------------------------------------
// Expression op Expression
//
// This includes all combinations of scalars, arrays, and fields.
//-----------------------------------------------------------------------------

template<class Op, class Left, class Right>
struct MakeFieldReturn<BinaryNode<Op, Left, Right> >
{
  // The node type.
  
  typedef BinaryNode<Op, Left, Right> Tree_t;

  // Deduce the template parameters of the expression engine we're building.
  
  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename ForEach<Tree_t, EvalLeaf<dim>, OpCombine>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;
  typedef typename ForEach<Tree_t, FarLeftTag, FarLeftTag>::
    Type_t::GeometryTag_t GeometryTag_t;
  
  // Construct the type of the Field we're making.

  typedef Field<GeometryTag_t, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a Field.

  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};


//-----------------------------------------------------------------------------
// Expression "?" Expression ":" Expression
//
// This includes all combinations of scalars, arrays, and fields.
//-----------------------------------------------------------------------------

template<class Op, class Left, class Middle, class Right>
struct MakeFieldReturn<TrinaryNode<Op, Left, Middle, Right> >
{
  // The node type.
  
  typedef TrinaryNode<Op, Left, Middle, Right> Tree_t;

  // Deduce the template parameters of the expression engine we're building.
  
  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename ForEach<Tree_t, EvalLeaf<dim>, OpCombine>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;
  typedef typename ForEach<Tree_t, FarLeftTag, FarLeftTag>::
    Type_t::GeometryTag_t GeometryTag_t;
  
  // Construct the type of the Field we're making.

  typedef Field<GeometryTag_t, T_t, ExpressionTag<Tree_t> > Expression_t;

  // This function turns the tree node into a Field.

  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};


#endif // POOMA_NEWFIELD_FIELDMAKERETURN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldMakeReturn.h,v $   $Author: sa_smith $
// $Revision: 1.1 $   $Date: 2000/08/08 17:31:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
