// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Field
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELD_H
#define POOMA_NEWFIELD_FIELD_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Field
//   - ties together the notions of field-category and mesh.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "Domain/CombineDomainOpt.h"
#include "Domain/NewDomain.h"
#include "Engine/ConstantFunctionEngine.h"
#include "Engine/Engine.h"
#include "Engine/EnginePatch.h"
#include "Engine/ExpressionEngine.h"
#include "Evaluator/Evaluator.h"
#include "PETE/PETE.h"
#include "Pooma/View.h"
#include "Utilities/PAssert.h"
#include "Utilities/RefCountedBlockPtr.h"

// NOTE:  The current order of includes puts FieldCreateLeaf after the
// operators files to work around a bug with template instantiation in KCC.

#include "NewField/FieldMakeReturn.h"
#include "NewField/FieldOperators.h"
#include "NewField/PoomaFieldOperators.h"
#include "NewField/VectorFieldOperators.h"
#include "NewField/FieldCreateLeaf.h"

#include "NewField/PrintField.h"

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

struct CompressibleBrick;

template<class GeometryTag, class T, class EngineTag>
class Field;

template<class GeometryTag, class T, class EngineTag> class FieldEngine;

template<class LTag, class EngineTag>
struct MultiPatch;

template<int Dim> struct NoGeometry;

struct POOMA_DEFAULT_ENGINE_TYPE;

template<class Subject> class SubFieldView;

template<class Subject, class Domain, bool SV>
struct View1Implementation;

//-----------------------------------------------------------------------------
// Prototypes for the assign function used to assign an expression to a Field.
//
// Prototypes defined here:
//   Field = Field
//   Field = Array
//   Field = scalar
///
// If you wish to have Field work with other types of objects on the right-
// hand side (for example, other classes that derive from Field), define
// extra assign() functions that take the following arguments:
//
//   assign(Field<Mesh,T,EngineTag>, yourclass, Operator)
//
// where "yourclass" is the class that you would like to work on the
// right-hand side in an expression with a Field on the left-hand side.
//-----------------------------------------------------------------------------
  
template<class GeometryTag, class T, class EngineTag,
  class GeometryTag2, class T2, class EngineTag2, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag,  T,  EngineTag> &lhs,
       const Field<GeometryTag2, T2, EngineTag2> &rhs,
       const Op &op);

template<class GeometryTag, class T, class EngineTag, 
 int Dim2, class T2, class EngineTag2, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag, T, EngineTag> &lhs, 
       const Array<Dim2, T2, EngineTag2> &rhs, const Op &op);

template<class GeometryTag, class T, class EngineTag, class T1, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag, T, EngineTag> &lhs, 
       const T1 &rhs, const Op &op);


//-----------------------------------------------------------------------------
// SubFieldView is used to implement the syntax f[i], which selects the
// ith SubField for field f.
//-----------------------------------------------------------------------------

struct SubFieldViewFunctorTag;

template<class GeometryTag, class T, class EngineTag>
class SubFieldView<Field<GeometryTag, T, EngineTag> > {
  
public:
  
  // Use it to construct the output field type.

  typedef Field<GeometryTag, T, EngineTag> Type_t;

  // The function that actually creates the view.
  
  inline static Type_t make(const Type_t &s, int iSubField)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(iSubField >= 0 && iSubField < s.numSubFields(),
        "Field::operator[] indexing error.");
#endif
      return Type_t(s, iSubField);
    }
};

template<class GeometryTag, class T, class Expr>
class SubFieldView<Field<GeometryTag, T, ExpressionTag<Expr> > > {
  
public:
  
  // Use it to construct the output field type.

  typedef Field<GeometryTag, T, ExpressionTag<Expr> > Subject_t;
  typedef 
    typename ForEach<Expr, SubFieldViewFunctorTag, TreeCombine>::Type_t 
      Expr_t;
  typedef Field<GeometryTag, T, ExpressionTag<Expr_t> > Type_t;

  // The function that actually creates the view.
  
  inline static Type_t make(const Subject_t &s, int iSubField)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(iSubField >= 0 && iSubField < s.numSubFields(),
        "Field::operator[] indexing error.");
#endif
      return Type_t(s, iSubField);
    }
};


//-----------------------------------------------------------------------------
// View1Implementation<Field, D, SV> specialization for indexing a field
// with a single domain. There is a single-valued version (SV == true)
// and a multi-valued version (SV == false).
//-----------------------------------------------------------------------------

// Single-valued version. Handles scalars and Locs.

template<class GeometryTag, class T, class EngineTag, class Domain>
struct View1Implementation<Field<GeometryTag, T, EngineTag>, Domain, true>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // The return types are pretty simple here.
  
  typedef typename Subject_t::Element_t ReadType_t;
  typedef typename Subject_t::ElementRef_t Type_t;

  template<class S1, class Combine>
  inline static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

  template<class S1, class S2, class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

  template<class S1, class S2, class S3,
    class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine()(s);
    }

  template<class S1, class Combine>
  inline static 
  ReadType_t makeRead(const Subject_t &f, const S1 &s1,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine().read(s);
    }

  template<class S1, class S2, class Combine>
  inline static 
  ReadType_t makeRead(const Subject_t &f,
	      const S1 &s1, const S2 &s2,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine().read(s);
    }

  template<class S1, class S2, class S3,
    class Combine>
  inline static 
  ReadType_t makeRead(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
    {
      PAssert(f.numSubFields() == 0);
      
      Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "Field view bounds error.");
#endif
      return f.engine().read(s);
    }
};

// Non-single-valued implementation. Works for general domains
// including Nodes and INodes.

// Use this little traits class to deduce the geometry tag for a view.
// It is always a NoGeometry unless the view is from an interval or
// an INode.

template<int Dim, class GeometryTag, class Domain>
struct NewGeometryTag
{
  typedef NoGeometry<Dim> Type_t;
};

template<int Dim, class GeometryTag>
struct NewGeometryTag<Dim, GeometryTag, Interval<Dim> >
{
  typedef GeometryTag Type_t;
};

template<int Dim, class GeometryTag>
struct NewGeometryTag<Dim, GeometryTag, INode<Dim> >
{
  typedef GeometryTag Type_t;
};

template<class GeometryTag, class T, class EngineTag, class Domain>
struct View1Implementation<Field<GeometryTag, T, EngineTag>, Domain, false>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename NewEngine<Engine_t, Domain>::Type_t NewEngine_t;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;
  
  // Deduce the new GeometryTag.
  
  typedef typename
    NewGeometryTag<NewEngine_t::dimensions, GeometryTag, Domain>::Type_t 
      NewGeometryTag_t;
  
  // The output types.
  
  typedef Field<NewGeometryTag_t, NewT_t, NewEngineTag_t> ReadType_t;
  typedef Field<NewGeometryTag_t, NewT_t, NewEngineTag_t> Type_t;

  template<class S1, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
	      "Field view bounds error.");
#endif

      return Type_t(f, s);
    }

  template<class S1, class S2, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const S2 &s2, const Combine &)
    {
      Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
	      "Field view bounds error.");
#endif

      return Type_t(f, s);
    }

  template<class S1, class S2, class S3,
    class Combine>
  static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
	      "Field view bounds error.");
#endif

      return Type_t(f, s);
    }

  template<class S1, class Combine>
  inline static 
  Type_t makeRead(const Subject_t &f, const S1 &s1,
	      const Combine &c)
    {
      return make(f, s1, c);
    }

  template<class S1, class S2, class Combine>
  inline static 
  Type_t makeRead(const Subject_t &f, const S1 &s1,
	      const S2 &s2, const Combine &c)
    {
      return make(f, s1, s2, c);
    }

  template<class S1, class S2, class S3,
    class Combine>
  inline static 
  Type_t makeRead(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &c)
    {
      return make(f, s1, s2, s3, c);
    }
};


//-----------------------------------------------------------------------------
// View1<Field, S1> specialization for indexing a field with a single domain.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class Sub1>
struct View1<Field<GeometryTag, T, EngineTag>, Sub1>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  // At some point, we need to fix NewDomain1; until then, use
  // the temporary version from NewDomain.h.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef TemporaryNewDomain1<Domain_t, Sub1> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return types.
  
  typedef typename Dispatch_t::ReadType_t ReadType_t;
  typedef typename Dispatch_t::Type_t Type_t;

  // The functions that create the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1)
    {
      return Dispatch_t::make(f, s1, Combine_t());
    }
  
  inline static
  ReadType_t makeRead(const Subject_t &f, const Sub1 &s1)
    {
      return Dispatch_t::makeRead(f, s1, Combine_t());
    }
};


//-----------------------------------------------------------------------------
// View1<Field, int> specialization for indexing a field with an int.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct View1<Field<GeometryTag, T, EngineTag>, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // The return types.
  
  typedef typename Subject_t::Element_t ReadType_t;
  typedef typename Subject_t::ElementRef_t Type_t;

  // The functions that do the indexing.

  inline static
  Type_t make(const Subject_t &f, int s1)
    {
      PAssert(f.numSubFields() == 0);

#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<1>(s1)),
        "Field view bounds error.");
#endif
      return f.engine()(s1);
    }

  inline static
  ReadType_t makeRead(const Subject_t &f, int s1)
    {
      PAssert(f.numSubFields() == 0);

#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<1>(s1)),
        "Field view bounds error.");
#endif
      return f.engine().read(s1);
    }
};


//-----------------------------------------------------------------------------
// View2<Field, S1, S2> specialization for indexing a field with two
// domains.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, 
  class Sub1, class Sub2>
struct View2<Field<GeometryTag, T, EngineTag>, Sub1, Sub2>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain2<Sub1, Sub2> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return types.
  
  typedef typename Dispatch_t::ReadType_t ReadType_t;
  typedef typename Dispatch_t::Type_t Type_t;

  // The functions that create the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2)
    {
      return Dispatch_t::make(f, s1, s2, Combine_t());
    }
  
  inline static
  ReadType_t makeRead(const Subject_t &f, const Sub1 &s1, const Sub2 &s2)
    {
      return Dispatch_t::makeRead(f, s1, s2, Combine_t());
    }
};


//-----------------------------------------------------------------------------
// View2<Field, int, int> specialization for indexing a field with two
// integers.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct View2<Field<GeometryTag, T, EngineTag>, int, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // The return types.
  
  typedef typename Subject_t::Element_t ReadType_t;
  typedef typename Subject_t::ElementRef_t Type_t;

  // The functions that do the indexing.

  inline static
  Type_t make(const Subject_t &f, int s1, int s2)
    {
      PAssert(f.numSubFields() == 0);
      
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<2>(s1, s2)),
        "Field view bounds error.");
#endif
      return f.engine()(s1, s2);
    }

  inline static
  ReadType_t makeRead(const Subject_t &f, int s1, int s2)
    {
      PAssert(f.numSubFields() == 0);
      
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<2>(s1, s2)),
        "Field view bounds error.");
#endif
      return f.engine().read(s1, s2);
    }
};


//-----------------------------------------------------------------------------
// View3<Field, S1, S2, S3> specialization for indexing a field with three
// domains.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, 
  class Sub1, class Sub2, class Sub3>
struct View3<Field<GeometryTag, T, EngineTag>, Sub1, Sub2, Sub3>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain3<Sub1, Sub2, Sub3> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return types.
  
  typedef typename Dispatch_t::ReadType_t ReadType_t;
  typedef typename Dispatch_t::Type_t Type_t;

  // The functions that create the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2, 
    const Sub3 &s3)
    {
      return Dispatch_t::make(f, s1, s2, s3, Combine_t());
    }
  
  inline static
  ReadType_t makeRead(const Subject_t &f, const Sub1 &s1, const Sub2 &s2, 
    const Sub3 &s3)
    {
      return Dispatch_t::makeRead(f, s1, s2, s3, Combine_t());
    }
};


//-----------------------------------------------------------------------------
// View3<Field, int, int, int> specialization for indexing a field with three
// integers.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct View3<Field<GeometryTag, T, EngineTag>, int, int, int>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;


  // The return types.
  
  typedef typename Subject_t::Element_t ReadType_t;
  typedef typename Subject_t::ElementRef_t Type_t;

  // The functions that do the indexing.
  
  inline static
  Type_t make(const Subject_t &f, int s1, int s2, int s3)
    {
      PAssert(f.numSubFields() == 0);
      
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<3>(s1, s2, s3)),
        "Field view bounds error.");
#endif
      return f.engine()(s1, s2, s3);
    }
  
  inline static
  ReadType_t makeRead(const Subject_t &f, int s1, int s2, int s3)
    {
      PAssert(f.numSubFields() == 0);
      
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<3>(s1, s2, s3)),
        "Field view bounds error.");
#endif
      return f.engine().read(s1, s2, s3);
    }
};


//-----------------------------------------------------------------------------
// Patch specialization for Field.
//-----------------------------------------------------------------------------

template<class Subject> struct Patch;

template<class GeometryTag, class T, class EngineTag>
struct Patch<Field<GeometryTag, T, EngineTag> >
{
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  typedef typename Subject_t::Engine_t OldEngine_t;
  typedef typename EngineFunctor<OldEngine_t, EnginePatch>::Type_t Engine_t;

  // We've assumed that GeometryTag and T are the same for the patch engine.

  typedef Field<GeometryTag, T, typename Engine_t::Tag_t> Type_t;

  inline static
  Type_t make(const Subject_t &f, int i)
    {
      PAssert(f.numSubFields() == 0);

      return Type_t(f, EnginePatch(i));
    }
};


//-----------------------------------------------------------------------------
// ComponentView specialization for Field. Implements views of the form
// f.comp(loc).
//-----------------------------------------------------------------------------

template<class Components, class Subject> struct ComponentView;

template<class Components>
class ComponentWrapper {
public:

  explicit ComponentWrapper(const Components &c) : c_m(c) { }
  
  const Components &components() const { return c_m; }
  
private:

  const Components &c_m;
};

template<class Components, class GeometryTag, class T, class EngineTag>
struct ComponentView<Components, Field<GeometryTag, T, EngineTag> >
{
  // Convenience typedef for the thing we're taking a component view of.
  
  typedef Field<GeometryTag, T, EngineTag> Subject_t;

  // Deduce the template parameters for the output type.
  
  typedef typename Subject_t::Element_t Element_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename ComponentAccess<Element_t, Components>::Element_t NewT_t;
  typedef CompFwd<Engine_t, Components> NewEngineTag_t;
  
  // The output type.
  
  typedef Field<GeometryTag, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Components &c)
    {
      return Type_t(f, ComponentWrapper<Components>(c));
    }
};


//-----------------------------------------------------------------------------
// Field.
//-----------------------------------------------------------------------------

template<class GeometryTag, 
         class T = POOMA_DEFAULT_ELEMENT_TYPE,
         class EngineTag = POOMA_DEFAULT_ENGINE_TYPE>
class Field {
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.
    
  // The specification type.
  
  typedef GeometryTag GeometryTag_t;

  // The type.
  
  typedef T T_t;
    
  // The engine tag.
  
  typedef EngineTag EngineTag_t;
  
  // This class.
  
  typedef Field<GeometryTag, T, EngineTag> This_t;
  
  // The field engine type.
  
  typedef FieldEngine<GeometryTag, T, EngineTag> FieldEngine_t;
  
  // The dimension (i.e., the number of indices required to select a point).
  
  enum { dimensions = FieldEngine_t::dimensions };
  
  // The engine type.
  
  typedef Engine<dimensions, T, EngineTag> Engine_t;

  // Element_t is the type of elements managed by this field's engine. 
  // ElementRef_t is the writable version.
  
  typedef typename Engine_t::Element_t Element_t;
  typedef typename Engine_t::ElementRef_t ElementRef_t;
  
  // Layout_t is the Engine's layout.
  
  typedef typename Engine_t::Layout_t Layout_t;
  
  // The types of the our domains. 

  typedef typename Engine_t::Domain_t Domain_t;

  
  //---------------------------------------------------------------------------
  // User-callable constructors. These ctors are meant to be called by users.

  // GeometryTag/centering/layout constructors. We use the specified mesh 
  // object to initialize our mesh and the layout to initialize 
  // the engines. Clearly, these must be synchronized. This is appropriate 
  // for multi-patch engines. We just store the centering.

  Field()
  : fieldEngine_m()
    { } 

  template<class I1>  
  explicit Field(const I1 &i1)
  : fieldEngine_m(i1)
    { } 

  template<class I1, class I2>  
  Field(const I1 &i1, const I2 &i2)
  : fieldEngine_m(i1, i2)
    { } 

  template<class I1, class I2, class I3>  
  Field(const I1 &i1, const I2 &i2, const I3 &i3)
  : fieldEngine_m(i1, i2, i3)
    { } 

  template<class I1, class I2, class I3, class I4>  
  Field(const I1 &i1, const I2 &i2, const I3 &i3, const I4 &i4)
  : fieldEngine_m(i1, i2, i3, i4)
    { } 

  // Copy constructor.
  
  Field(const This_t &model)
  : fieldEngine_m(model.fieldEngine())
    { }

  
  //---------------------------------------------------------------------------
  // Internal POOMA constructors. These ctors are used internally by POOMA.
  // They are not really meant to be called by users.

  // Model-initializer constructor. Used by SubFieldView and 
  // View1Implementation above and by MakeFieldReturn in FieldCreateLeaf.h.

  template<class GT2, class T2, class ET2, class Initializer>
  Field(const Field<GT2, T2, ET2> &model, const Initializer &i)
  : fieldEngine_m(model.fieldEngine(), i)
    { }

  // Centering-Model constructor. This version differs from the one above
  // in that it constructs a field with uninitialized engines with a new
  // centering.  The new field gets its mesh and domain information from
  // the model field.

  template<class Centering, class GT2, class T2, class ET2>
  Field(const Centering &baseInit, const Field<GT2, T2, ET2> &model)
    : fieldEngine_m(baseInit, model.fieldEngine())
  { }


  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~Field() { }


  //---------------------------------------------------------------------------
  // Accessors.

  inline const Engine_t &engine() const
    {
      return fieldEngine_m.engine();
    }

  inline const FieldEngine_t &fieldEngine() const
    {
      return fieldEngine_m;
    }
    
  inline FieldEngine_t &fieldEngine()
    {
      return fieldEngine_m;
    }
    
  inline int numSubFields() const
    {
      return fieldEngine_m.numSubFields();
    }
        
  inline const Domain_t &physicalCellDomain() const
    {
      return fieldEngine_m.physicalCellDomain();
    }
        
  inline Domain_t totalCellDomain() const
    {
      return fieldEngine_m.totalCellDomain();
    }

  Domain_t physicalDomain(int iSubfield) const
    {
      return fieldEngine_m.physicalDomain(iSubfield);
    }

  Domain_t totalDomain(int iSubfield) const
    {
      return fieldEngine_m.totalDomain(iSubfield);
    }

  Domain_t physicalDomain() const
    {
      return fieldEngine_m.physicalDomain();
    }

  Domain_t totalDomain() const
    {
      return fieldEngine_m.totalDomain();
    }

  Domain_t domain() const
    {
      return fieldEngine_m.physicalDomain();
    }


  //---------------------------------------------------------------------------
  // Instruct the field to make its own copy of its data.
  // Recursively call ourself with subfield views of this field. When we're
  // through, tell the fieldEngine to make a distinct copy of itself.

  void makeOwnCopy()
    {
      if (numSubFields() == 0)
        {
          // Make a distinct copy of the fieldEngine.
          
          fieldEngine_m.makeOwnCopy(*this);
        }
      else
        {
          for (int i = 0; i < n; i++)
            (*this)[i].makeOwnCopy();
        }
    }
      
  
  //---------------------------------------------------------------------------
  // Sub-field view creation function.
  // A field consists of (potentially) several sub-fields. This function
  // returns a view of one of these.

  inline typename SubFieldView<This_t>::Type_t
  operator[](int iSubfield) const
    {
      typedef SubFieldView<This_t> Ret_t;
      return Ret_t::make(*this, iSubfield);
    }


  //---------------------------------------------------------------------------
  // View-creation operations. These operator() and read() functions take 
  // zero or more sub-domains, which combine to form a domain with 
  // dimensionality identical to the rank of the field. The zero argument 
  // version returns a view of the physical domain and the 'All'-suffixed
  // versions return a view of the total domain.

  inline typename View1<This_t, Domain_t>::ReadType_t 
  read() const
    {
      typedef View1<This_t, Domain_t> Ret_t;
      return Ret_t::makeRead(*this, physicalDomain());
    }

  inline typename View1<This_t, Domain_t>::ReadType_t 
  readAll() const
    {
      typedef View1<This_t, Domain_t> Ret_t;
      return Ret_t::makeRead(*this, totalDomain());
    }

  template<class Sub1> 
  inline typename View1<This_t, Sub1>::ReadType_t 
  read(const Sub1 &s1) const
    {
      typedef View1<This_t, Sub1> Ret_t;
      return Ret_t::makeRead(*this, s1);
    }

  template<class Sub1, class Sub2> 
  inline typename View2<This_t, Sub1, Sub2>::ReadType_t 
  read(const Sub1 &s1, const Sub2 &s2) const
    {
      typedef View2<This_t, Sub1, Sub2> Ret_t;
      return Ret_t::makeRead(*this, s1, s2);
    }

  template<class Sub1, class Sub2, class Sub3> 
  inline typename View3<This_t, Sub1, Sub2, Sub3>::ReadType_t 
  read(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
    {
      typedef View3<This_t, Sub1, Sub2, Sub3> Ret_t;
      return Ret_t::make(*this, s1, s2, s3);
    }

  inline typename View1<This_t, Domain_t>::Type_t 
  operator()() const
    {
      typedef View1<This_t, Domain_t> Ret_t;
      return Ret_t::make(*this, physicalDomain());
    }

  inline typename View1<This_t, Domain_t>::Type_t 
  all() const
    {
      typedef View1<This_t, Domain_t> Ret_t;
      return Ret_t::make(*this, totalDomain());
    }

  template<class Sub1> 
  inline typename View1<This_t, Sub1>::Type_t 
  operator()(const Sub1 &s1) const
    {
      typedef View1<This_t, Sub1> Ret_t;
      return Ret_t::make(*this, s1);
    }

  template<class Sub1, class Sub2> 
  inline typename View2<This_t, Sub1, Sub2>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2) const
    {
      typedef View2<This_t, Sub1, Sub2> Ret_t;
      return Ret_t::make(*this, s1, s2);
    }

  template<class Sub1, class Sub2, class Sub3> 
  inline typename View3<This_t, Sub1, Sub2, Sub3>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
    {
      typedef View3<This_t, Sub1, Sub2, Sub3> Ret_t;
      return Ret_t::make(*this, s1, s2, s3);
    }


  //---------------------------------------------------------------------------
  // Component-forwarding functions. These work quite similar to the ones from
  // Array except we produce a Field with the same GeometryTag.

  inline typename ComponentView<Loc<1>, This_t>::Type_t
  comp(const int &i1) const
  {
    return ComponentView<Loc<1>, This_t>::make(*this, Loc<1>(i1));
  }

  inline typename ComponentView<Loc<2>, This_t>::Type_t
  comp(int i1, int i2) const
  {
    return ComponentView<Loc<2>, This_t>::make(*this, Loc<2>(i1, i2));
  }

  template<class Components>
  inline typename ComponentView<Components, This_t>::Type_t
  comp(const Components &loc) const
  {
    return ComponentView<Components, This_t>::make(*this, loc);
  }


  //---------------------------------------------------------------------------
  // Patch accessor functions returns the i'th patch.

  inline typename Patch<This_t>::Type_t
  patchLocal(EnginePatch::PatchID_t i) const
    {
      return Patch<This_t>::make(*this, i);
    }


  //---------------------------------------------------------------------------
  // Copy assignment operators. We pack this assignment expression into a
  // PETE binary expression tree node and then use this to construct an
  // array with an expression engine. We then pass this on to an evaluator,
  // which handles the computation. The first three versions handle assigning
  // Arrays and ConstArrays to Arrays and the fourth one handles assigning
  // scalars.

  This_t &operator=(const This_t &rhs)
    {
      assign(*this, rhs, OpAssign());
      return *this;
    }

  const This_t &operator=(const This_t &rhs) const
    {
      return assign(*this, rhs, OpAssign());
    }

  template<class T1>
  const This_t &operator=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpAssign());
    }


  //---------------------------------------------------------------------------
  // Op-assignment operators. 

  // Addition.

  template<class T1>
  const This_t &operator+=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpAddAssign());
    }

  // Subtraction.

  template<class T1>
  const This_t &operator-=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpSubtractAssign());
    }

  // Multiplication.

  template<class T1>
  const This_t &operator*=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpMultiplyAssign());
    }

  // Division.

  template<class T1>
  const This_t &operator/=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpDivideAssign());
    }

  // Modulus.

  template<class T1>
  const This_t &operator%=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpModAssign());
    }

  // Bitwise-Or.

  template<class T1>
  const This_t &operator|=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpBitwiseOrAssign());
    }

  // Bitwise-And.

  template<class T1>
  const This_t &operator&=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpBitwiseAndAssign());
    }

  // Bitwise-Xor.

  template<class T1>
  const This_t &operator^=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpBitwiseXorAssign());
    }

  // Left shift.

  template<class T1>
  const This_t &operator<<=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpLeftShiftAssign());
    }

  // Right shift.

  template<class T1>
  const This_t &operator>>=(const T1 &rhs) const
    {
      return assign(*this, rhs, OpRightShiftAssign());
    }


  //---------------------------------------------------------------------------
  // Automatic update support.
  
  template<class Category>         
  void addUpdater(const Category &bc)
    {
      if (numSubFields() == 0)
        fieldEngine_m.updaters().addUpdater(*this, bc);
      else
        {
          for (int i = 0; i < numSubFields(); i++)
            (*this)[i].addUpdater(bc);
        }
    }

  template<class UpdaterIntitializer>  
  void addUpdaters(const UpdaterIntitializer &init)
    {
      init(*this);
    }
  
  void removeUpdaters()
    {
      if (numSubFields() == 0)
        fieldEngine_m.updaters().erase();
      else
        {
          for (int i = 0; i < numSubFields(); i++)
            (*this)[i].removeUpdaters();
        }
    }
  
  void update(bool makeDirty = false) const
    {
      if (numSubFields() == 0)
        {
          if (makeDirty)
            fieldEngine_m.updaters().setDirty();
          fieldEngine_m.updaters().notifyPreRead();
        }
      else
        {
          for (int i = 0; i < numSubFields(); i++)
            (*this)[i].update(makeDirty);
        }
    }
  
  void setDirty() const
    {
      if (numSubFields() == 0)
        {
          fieldEngine_m.updaters().setDirty();
        }
      else
        {
          for (int i = 0; i < numSubFields(); i++)
            (*this)[i].setDirty();
        }
    }
  
  void clearDirty() const
    {
      if (numSubFields() == 0)
        {
          fieldEngine_m.updaters().clearDirty();
        }
      else
        {
          for (int i = 0; i < numSubFields(); i++)
            (*this)[i].clearDirty();
        }
    }


private:

  FieldEngine_t fieldEngine_m;

};


//----------------------------------------------------------------------
// Set up a little traits class that distinguishes between OpAssign and
// other assignment operators that read the LHS.
//----------------------------------------------------------------------

template<class Op>
struct AssignOpReadWriteTraits
{
  enum { readLHS = true };
};

template<>
struct AssignOpReadWriteTraits<OpAssign>
{
  enum { readLHS = false };
};


//----------------------------------------------------------------------
// Apply the ConformTag to the leaves of the tree.
// Check to see if a given Field conforms.
//----------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, int Dim>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, ConformTag<Dim> >
{
  typedef bool Type_t;
  static Type_t apply1(const Interval<Dim> &d, 
    const ConformTag<Dim> &ct)
    {
      return conforms(d, ct);
    }
  template<int Dim2>
  static Type_t apply1(const Interval<Dim2> &d, 
    const ConformTag<Dim> &ct)
    {
      return false;
    }
  static Type_t apply(const Field<GeometryTag, T, EngineTag> &f,
    const ConformTag<Dim> &ct)
    {
      return apply1(f.physicalDomain(), ct);
    }
};


//----------------------------------------------------------------------
// This specialization of LeafFunctor is used to pass the 
// DataObjectRequest functor down into the FieldEngine. The default 
// behavior, given in the functor below, is to just pass it on to the 
// fieldEngine's engine.
//----------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class RequestType>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>,
  DataObjectRequest<RequestType> >
{
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  typedef typename Subject_t::FieldEngine_t FieldEngine_t;
  typedef LeafFunctor<FieldEngine_t, DataObjectRequest<RequestType> > 
    LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;
  enum { dataObject = LeafFunctor_t::dataObject };
  
  inline static
  Type_t apply(const Subject_t &f,
	           const DataObjectRequest<RequestType> &functor)
    {
      return LeafFunctor_t::apply(field.fieldEngine(), functor);
    }
};

template<class GeometryTag, class T, class EngineTag, class RequestType>
struct LeafFunctor<FieldEngine<GeometryTag, T, EngineTag>,
  DataObjectRequest<RequestType> >
{
  typedef typename FieldEngine<GeometryTag, T, EngineTag>::Engine_t 
    Engine_t;
  enum { dataObject = Engine_t::dataObject };
  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;
  inline static
  Type_t apply(const FieldEngine<GeometryTag, T, EngineTag> &f,
	           const DataObjectRequest<RequestType> &functor)
    {
      return DataObjectApply<dataObject>::apply(f.engine(), functor);
    }
};


//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to get the domain type or the
// (zero-based) domain itself from a Field. Used only by Expression-Engine.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, DomainFunctorTag>
{
  typedef typename Field<GeometryTag, T, EngineTag>::Domain_t Type_t;

  inline static Type_t apply(const Field<GeometryTag, T, EngineTag> &f, 
    const DomainFunctorTag &)
    {
      // Return zero-based domain.
      
      return f.physicalDomain() - f.physicalDomain().firsts();
    }
};


//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to pass the EngineApply functor
// down into the FieldEngine. The default behavior, given in the functor
// below, is to just pass it on to the fieldEngine's engine.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class Tag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, EngineApply<Tag> >
{
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  typedef typename Subject_t::FieldEngine_t FieldEngine_t;
  typedef LeafFunctor<FieldEngine_t, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  inline static
  Type_t apply(const Subject_t &field, 
	       const EngineApply<Tag> &tag)
    {
      return LeafFunctor_t::apply(field.fieldEngine(), tag);
    }
};

template<class GeometryTag, class T, class EngineTag, class Tag>
struct LeafFunctor<FieldEngine<GeometryTag, T, EngineTag>, EngineApply<Tag> >
{
  typedef FieldEngine<GeometryTag, T, EngineTag> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  inline static
  Type_t apply(const Subject_t &fieldEngine, 
	           const EngineApply<Tag> &tag)
    {
      return LeafFunctor_t::apply(fieldEngine.engine(), tag);
    }
};


//-----------------------------------------------------------------------------
// Handle general engine functor requests.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class Tag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, EngineFunctorTag<Tag> >
{
  typedef typename Field<GeometryTag,T,EngineTag>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t,Tag>::Type_t Type_t;
  inline static
  Type_t apply(const Field<GeometryTag, T, EngineTag> &field,
	       const EngineFunctorTag<Tag> &tag)
  {
    return EngineFunctor<Engine_t,Tag>::apply(field.engine(), tag.tag());
  }
};


//---------------------------------------------------------------------------
// A specialization of EngineFunctor for field.
//---------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class Tag>
struct EngineFunctor<Field<GeometryTag, T, EngineTag>, Tag>
{
  typedef typename Field<GeometryTag, T, EngineTag>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t, Tag>::Type_t Type_t;

  inline static 
  Type_t apply(const Field<GeometryTag, T, EngineTag> &field,
	           const Tag &tag)
    {
      return engineFunctor(field.engine(), tag);
    }
};


//-----------------------------------------------------------------------------
// This version of LeafFunctor is used by Expression-Engines to used to 
// evaluate a Field using indices. 
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, int Dim>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, EvalLeaf<Dim> >
{
  typedef typename Field<GeometryTag, T, EngineTag>::Element_t Type_t;
  inline static
  Type_t apply(const Field<GeometryTag, T, EngineTag> &f, 
    const EvalLeaf<Dim> &t) 
    {
      return t.eval(f.engine());
    }
};


//-----------------------------------------------------------------------------
// These leaf functor specializations are used to notify a field or expression
// that it is going to be read and, therefore, needs to update itself. 
//
// The first LeafFunctor represents default behavior, which is to do nothing. 
// The second handles fields other than those with expression-engines by simply
// calling update(). The third passes the tag to the leaves.
//
// Fields with engines that store internal fields AND don't copy those
// fields' updaters to its list must provide a specialization to get the 
// internal fields to update.
//
// NOTE: we don't use the EngineApply machinery here because this really
// operate on the engines.
//
//-----------------------------------------------------------------------------

struct PerformUpdateTag
{
  POOMA_PURIFY_CONSTRUCTORS(PerformUpdateTag)
};

template<class Node>
struct LeafFunctor<Node, PerformUpdateTag>
{
  typedef int Type_t;

  inline static
  Type_t apply(const Node &, const PerformUpdateTag &)
    {
      return 0;
    }
};


template<class GeometryTag, class T, class EngineTag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, 
  PerformUpdateTag>
{
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  typedef int Type_t;

  inline static
  Type_t apply(const Subject_t &f, const PerformUpdateTag &)
    {
      f.update();
      return 0;
    }
};

template<class GeometryTag, class T, class Expr>
struct LeafFunctor<Field<GeometryTag, T, ExpressionTag<Expr> >, 
  PerformUpdateTag>
{
  typedef Field<GeometryTag, T, ExpressionTag<Expr> > Subject_t;
  typedef int Type_t;

  inline static
  Type_t apply(const Subject_t &f, const PerformUpdateTag &tag)
    {
      forEach(f.engine().expression(), tag, NullCombine());
      return 0;
    }
};


//-----------------------------------------------------------------------------
// This version of LeafFunctor is used to determine the type resulting from a
// sub-field view. 
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, SubFieldViewFunctorTag>
{
  typedef Field<GeometryTag, T, EngineTag> Type_t;
};

template<class T>
struct LeafFunctor<Scalar<T>, SubFieldViewFunctorTag>
{
  typedef Scalar<T> Type_t;
};


//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to apply a view (subsetting) 
// operation to a Field. The domain will always be zero-based since this
// is used only by Expression-Engine. This is why we add the firsts() to
// the domain.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class Domain>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, ViewFunctorTag<Domain> >
{
  typedef typename View1<Field<GeometryTag, T, EngineTag>, Domain>::Type_t 
    Type_t;
};


//-----------------------------------------------------------------------------
// Function for removing updaters from a Field.
//-----------------------------------------------------------------------------

namespace Pooma {

  template<class GeometryTag, class T, class EngineTag>
  void removeUpdaters(Field<GeometryTag, T, EngineTag> &f)
    {
      f.removeUpdaters();
    }
}


//-----------------------------------------------------------------------------
// Overload the << operator to print a Field to a stream.  This
// uses the 'PrintField' class to perform the formatting of the data.
// It will create a default printer, print out the field with it, and
// return the stream object.
//-----------------------------------------------------------------------------

template <class GeometryTag, class T, class EngineTag>
std::ostream &operator<<(std::ostream &o, 
  const Field<GeometryTag, T, EngineTag> &cf)
{
  Pooma::blockAndEvaluate();
  PrintField().print(o, cf);
  return o;
}

template <class GeometryTag, class T, class EngineTag>
std::fstream &operator<<(std::fstream &f, 
  const Field<GeometryTag, T, EngineTag> &cf)
{
  Pooma::blockAndEvaluate();
  PrintField().print(f, cf);
  return f;
}


//-----------------------------------------------------------------------------
// assign() function for Field assign-op array.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, 
 int Dim2, class T2, class EngineTag2, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag, T, EngineTag> &lhs, 
       const Array<Dim2, T2, EngineTag2> &rhs, const Op &op)
{
  int nsf = lhs.numSubFields();
  
  if (nsf != 0)
    {
      // We need to take sub-field views in order to get at all of the
      // internal fields.
      
      for (int i = 0; i < nsf; i++)
        assign(lhs[i], rhs, op);
    }
  else
    {
      if (AssignOpReadWriteTraits<Op>::readLHS)
        lhs.update();
     
      // Evaluate. 

      Evaluator<MainEvaluatorTag>().evaluate(lhs, op, rhs);

      // Having done the evaluation, we need to notify the LHS
      // that we've just written.
  
      lhs.setDirty();
    }
        
  return lhs;
}


//-----------------------------------------------------------------------------
// assign() function for Field assign-op Field.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag,
  class GeometryTag2, class T2, class EngineTag2, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag, T, EngineTag> &lhs,
       const Field<GeometryTag2, T2, EngineTag2> &rhs,
       const Op &op)
{
  int nsf = lhs.numSubFields();
  
  if (nsf != 0)
    {
      // We need to take sub-field views in order to get at all of the
      // internal fields.
      
      for (int i = 0; i < nsf; i++)
        assign(lhs[i], rhs[i], op);
    }
  else
    {
      // We need to notify the RHS and, when not doing pure assignment, the
      // LHS that we are getting ready to read.
      
      forEach(rhs, PerformUpdateTag(), NullCombine());
      
      if (AssignOpReadWriteTraits<Op>::readLHS)
        lhs.update();
  
      // Evaluate.

      Evaluator<MainEvaluatorTag>().evaluate(lhs, op, rhs);

      // Having done the evaluation, we need to notify the LHS
      // that we've just written.
  
      lhs.setDirty();
    }
        
  return lhs;
}


//-----------------------------------------------------------------------------
// assign() function for Field assign-op scalar.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag, class T1, class Op>
const Field<GeometryTag, T, EngineTag> &
assign(const Field<GeometryTag, T, EngineTag> &lhs, const T1 &rhs, const Op &op)
{
  int nsf = lhs.numSubFields();
  
  if (nsf != 0)
    {
      // We need to take sub-field views in order to get at all of the
      // internal fields.
      
      for (int i = 0; i < nsf; i++)
        assign(lhs[i], rhs, op);
    }
  else
    {
      if (AssignOpReadWriteTraits<Op>::readLHS)
        lhs.update();

      // Make an array out of the scalar.

      typedef Field<GeometryTag, T, EngineTag> LHS_t;
      Array<LHS_t::dimensions, T1, ConstantFunction>  rhsExpr(lhs.physicalDomain());
      rhsExpr.engine().setConstant(rhs);
     
      // Evaluate. 

      Evaluator<MainEvaluatorTag>().evaluate(lhs, op, rhsExpr);

      // Having done the evaluation, we need to notify the LHS
      // that we've just written.
  
      lhs.setDirty();
    }
        
  return lhs;
}


//-----------------------------------------------------------------------------
// Compute the fraction of the total domain that is currently compressed.
//
// This is only a sensible thing to do if there are no subfields, hence the
// assertion.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class LTag>
inline double
compressedFraction(
  const Field<GeometryTag, T, MultiPatch<LTag,CompressibleBrick> > &f)
{
  PAssert(f.numSubFields() == 0);
  return compressedFraction(f.engine());
}


//-----------------------------------------------------------------------------
// (Try to) compress all the patches of the Field. Only need to do work with
// multipatch engines.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class LTag>
void
compress(Field<GeometryTag, T, MultiPatch<LTag,CompressibleBrick> > &)
{
  if (numSubFields() == 0)
    {
      compress(f.engine());
    }
  else
    {
      for (int i = 0; i < numSubFields(); i++)
        compress(f[i]);
    }
}


//-----------------------------------------------------------------------------
// Manually uncompress all the patches of the Field. Only need to do work with
// multipatch engines.
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class LTag>
void
uncompress(Field<GeometryTag, T, MultiPatch<LTag,CompressibleBrick> > &f)
{
  if (numSubFields() == 0)
    {
      upcompress(f.engine());
    }
  else
    {
      for (int i = 0; i < numSubFields(); i++)
        upcompress(f[i]);
    }
}

#endif // POOMA_NEWFIELD_FIELD_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Field.h,v $   $Author: sa_smith $
// $Revision: 1.5 $   $Date: 2000/08/08 17:31:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
