// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// UniformRectilinearMesh test 1: Basic stuff, including MeshView.
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the total domain.
  
  Interval<1> I(3), J(4);
  
  Vector<2> origin;
  Vector<2> spacings;
  for (int d = 0; d < 2; d++) {
    origin(d) = d;
    spacings(d) = d + 1;
  }
  UniformRectilinearMesh<2, Cartesian<2> > mesh(I, J, origin, spacings);
  tester.out() << mesh << std::endl;

  Interval<2> pd(3, 4), td(Interval<1>(-1,3), Interval<1>(-2,5));
  Interval<2> pcd(2, 3), tcd(Interval<1>(-1,2), Interval<1>(-2,4));
  tester.check("mesh PD", mesh.physicalDomain(), pd); 
  tester.check("mesh TD", mesh.totalDomain(), td);
  tester.check("mesh PCD", mesh.physicalCellDomain(), pcd);
  tester.check("mesh TCD", mesh.totalCellDomain(), tcd);

  tester.check("mesh PD volume", mesh.totalVolumeOfCells(), 12.0);
  tester.check("mesh TD volume", 
    mesh.totalVolumeOfCells(mesh.totalCellDomain()), 56.0);

  tester.check("mesh origin", mesh.vertexPositions().read(0,0), origin);
  tester.check("mesh vertex", mesh.vertexPositions().read(3,5), 
    Vector<2>(3,11));
  tester.check("mesh surf normals", mesh.cellSurfaceNormals().read(0,0)(2), 
    Vector<2>(0,-1));
  
  tester.check("mesh NV1", mesh.nearestVertex(Vector<2>(3,9.5)), 
    Loc<2>(3,4));
  tester.check("mesh NV2", mesh.nearestVertex(Vector<2>(1,9)), 
    Loc<2>(1,4));
  tester.check("mesh cell cont", mesh.cellContaining(origin), Loc<2>(0,0));
  
  Interval<2> V(Interval<1>(1,3), Interval<1>(-1,4));
  MeshView<UniformRectilinearMesh<2, Cartesian<2> > > mview(mesh, V);
  tester.out() << mview << std::endl;

  tester.check("mview PD", mview.physicalDomain(), V - V.firsts());
  tester.check("mview TD", mview.physicalDomain(), mview.totalDomain());
  tester.check("mview PCD", mview.physicalCellDomain(), 
    growRight(V - V.firsts(), -1));
  tester.check("mview TCD", mview.totalCellDomain(), 
    mview.physicalCellDomain());

  tester.check("mview PD volume", mview.totalVolumeOfCells(), 20.0);
  tester.check("mview TD volume", 
    mview.totalVolumeOfCells(mview.totalCellDomain()), 20.0);

  tester.check("mview origin", 
    mview.vertexPositions().read(0,0), mesh.vertexPositions().read(1,-1));
  tester.check("mview surf normals", 
    mview.cellSurfaceNormals().read(0,0)(2), Vector<2>(0,-1));

  tester.check("mview NV", 
    mview.nearestVertex(Vector<2>(2,2)), Loc<2>(1,2));
  tester.check("mview VB", 
    mview.vertexBelow(Vector<2>(1,9)), Loc<2>(0,5));
  tester.check("mview cell cont", 
    mview.cellContaining(Vector<2>(3,2)), Loc<2>(2,1));
  
  Interval<2> W(Interval<1>(1,2), Interval<1>(2,4));
  MeshView<UniformRectilinearMesh<2, Cartesian<2> > > mview2(mview, W);
  tester.out() << mview2 << std::endl;

  tester.check("mview2 PD", mview2.physicalDomain(), W - W.firsts());
  tester.check("mview2 TD", mview2.physicalDomain(), mview2.totalDomain());
  tester.check("mview2 PCD", 
    mview2.physicalCellDomain(), growRight(W - W.firsts(), -1));
  tester.check("mview2 TCD", 
    mview2.totalCellDomain(), mview2.physicalCellDomain());

  tester.check("mview2 PD volume", mview2.totalVolumeOfCells(), 4.0);
  tester.check("mview2 TD volume", 
    mview2.totalVolumeOfCells(mview2.totalCellDomain()), 4.0);

  tester.check("mview2 origin", 
    mview2.vertexPositions().read(0,0), mesh.vertexPositions().read(2,1));
  tester.check("mview2 surf normals", 
    mview2.cellSurfaceNormals().read(0,0)(2), Vector<2>(0,-1));

  tester.check("mview2 NV",
    mview2.nearestVertex(Vector<2>(2,4)), Loc<2>(0,1));
  tester.check("mview2 VB",
    mview2.vertexBelow(Vector<2>(2,6)), Loc<2>(0,1));
  tester.check("mview2 cell cont",
    mview2.cellContaining(Vector<2>(3,4)), Loc<2>(1,0));
  
  int ret = tester.results("urm_test1");
  Pooma::finalize();
    
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: urm_test1.cpp,v $   $Author: julianc $
// $Revision: 1.8 $   $Date: 2000/07/20 23:38:23 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
