// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_LAYOUT_GRIDLAYOUT_H
#define POOMA_LAYOUT_GRIDLAYOUT_H

//-----------------------------------------------------------------------------
// Classes: 
//   GridLayout<Dim>
//   GridLayoutView<Dim, Dim2>
//   GridTag
//   MultiPatchLayoutTraits<GridTag,Dim>
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Overview:
//
//   GridLayout<Dim>
//     - Layout class that breaks Dim-dimensional domain into
//       sub-domains arranged in a Dim-dimensional grid, where the
//       sub-domains are defined by a set of intervals along each axis. 
//   GridLayoutView<Dim, Dim2>
//     - view of a GridLayout
//   GridTag
//     - tag used to specialize MultiPatchLayoutTraits
//   MultiPatchLayoutTraits<GridTag,Dim>
//     - traits class used by MultiPatch-engine to determine layout type.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Layout/MultiPatchLayoutTraits.h"
#include "Layout/INode.h"
#include "Layout/TouchesConstruct.h"
#include "Layout/GuardLayers.h"
#include "Layout/DynamicEvents.h"
#include "Partition/UniformGridPartition.h"
#include "Partition/GridPartition.h"
#include "Partition/ContextMapper.h"
#include "Domain/Interval.h"
#include "Domain/Grid.h"
#include "Domain/Contains.h"
#include "Domain/Intersect.h"
#include "Domain/NewDomain.h"
#include "Domain/SliceRange.h"
#include "Domain/DomainMap.h"
#include "Utilities/DerefIterator.h"
#include "Utilities/ViewIndexer.h"
#include "Utilities/Observable.h"
#include "Utilities/Observer.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"
#include "Utilities/Unique.h"
#include "Utilities/PAssert.h"

#include "Layout/LayoutBase.h"

#include <vector>
#include <iosfwd>


///////////////////////////////////////////////////////////////////////////////
// namespace Pooma {

//============================================================
// Forward declarations
//============================================================

template <int Dim> class GridLayoutData;
template <int Dim> class GridLayout;

template <int Dim, int Dim2> class GridLayoutViewData;
template <int Dim, int Dim2> class GridLayoutView;


//-----------------------------------------------------------------------------
// Full Description:
// GridTag
//
// GridTag class.
//-----------------------------------------------------------------------------

struct GridTag { };


//-----------------------------------------------------------------------------
// Full Description:
// MultiPatchLayoutTraits<GridTag,Dim>
//
// Specialization of MultiPatchLayoutTraits for GridLayout.
//-----------------------------------------------------------------------------

template <int Dim>
struct MultiPatchLayoutTraits<GridTag,Dim>
{
  typedef GridLayout<Dim> Layout_t;
  
  template <int ViewDim>
  struct View
  {
    typedef GridLayoutView<ViewDim,Dim> Layout_t;
  };
};


//-----------------------------------------------------------------------------
//
// Full Description:
// GridLayoutData<Dim>
//
// Holds the data for a GridLayout. That class has a ref counted
// instance of this class
//-----------------------------------------------------------------------------

template<int Dim>
class GridLayoutData 
  : public LayoutBaseData<Dim>,
    public RefCounted,
    public Observable< GridLayoutData<Dim> >
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  // General public typedefs.

  typedef GridLayoutData<Dim>                  This_t;
  typedef Observable<This_t>                   Observable_t;
  typedef Interval<Dim>                        Domain_t;
  typedef Interval<Dim>                        BaseDomain_t;
  typedef int                                  Context_t;
  typedef Unique::Value_t                      ID_t;
  typedef Node<Domain_t>                       Value_t;
  typedef std::vector<Value_t *>               List_t;
  typedef GuardLayers<Dim>                     GuardLayers_t;    
  typedef int                                  AxisIndex_t;
  typedef typename DynamicEvents::PatchID_t    PatchID_t;
  typedef typename DynamicEvents::CreateSize_t CreateSize_t;

  typedef typename LayoutBaseData<Dim>::GCFillInfo_t GCFillInfo_t;

  typedef typename std::vector<GCFillInfo_t>::const_iterator FillIterator_t;

  // Enumerations.

  enum { dimensions = Dim };
  enum { repartitionEvent = 1 };
  enum { dynamic = false };


  //============================================================
  // Constructors
  //============================================================

 
// Default constructor for GridLayoutData, this sets up this object
// to look like an "empty" layout, with no patches and an empty domain
// with no guard cells.  The "initialize" method can be used to change
// it to a different state.

  GridLayoutData();


  // If we specify a partitioner, then we'll get the guard-cell sizes
  // from it.

  template <class Partitioner>
  GridLayoutData(const Domain_t &gdom, 
		 const Partitioner &gpar,
		 const ContextMapper<Dim> & cmap);
  

  template <class Partitioner>
  GridLayoutData(const Grid<Dim> &gdom, 
		 const Partitioner &gpar,
                 const ContextMapper<Dim> &cmap);


  //============================================================
  // Destructor
  //============================================================

  // This is only called when all references to this data go away, in
  // which case we need to delete our nodes. The Observable destructor
  // will broadcast messages up to all observers of the Layout.

  ~GridLayoutData()
    {
      for (typename List_t::iterator a = all_m.begin(); a != all_m.end(); ++a)
	delete (*a);
    } 
  
  //============================================================
  // Mutators
  //============================================================

  // Initialize this object by invoking the partitioner and setting
  // up the domains and guard cell filling info.  This can be called
  // after using the default constructor.



  template <class Partitioner>
  void initialize(const Domain_t &gdom, 
		  const Partitioner &gpar,
		  const ContextMapper<Dim> &cmap);

  template <class Partitioner>
  inline void initialize(const Grid<Dim> &gdom, 
			 const Partitioner &gpar,
                         const ContextMapper<Dim> &cmap);

  // Used by the I/O or data management system to initialize the layout based
  // on detailed state information previously stored.

  void initialize(const Domain_t& idom,
		  const List_t& nodes,
		  const Loc<Dim>& blocks,
		  bool hasIG, bool hasEG,
		  const GuardLayers_t& ig,
		  const GuardLayers_t& eg);

  //============================================================
  // Accessors
  //============================================================

  inline const Loc<Dim> &blocks() const
    {
      return blocks_m;
    }

  inline bool dirty() const
    {
      return dirtyLayout_m;
    }

  // Accessors for getting the global ID of the patch containing
  // a particular element.

  int globalID(const Loc<Dim> &loc) const;
  int globalID(int) const;
  int globalID(int,int) const;
  int globalID(int,int,int) const;
  int globalID(int,int,int,int) const;
  int globalID(int,int,int,int,int) const;
  int globalID(int,int,int,int,int,int) const;
  int globalID(int,int,int,int,int,int,int) const;


  // Iterators into the fill list. These are MultiPatch's interface to
  // the information needed to fill guard cells, which is cached here
  // in the layout.
   
  FillIterator_t beginFillList() const
    {
      return gcFillList_m.begin();
    }
   
  FillIterator_t endFillList() const
    {
      return gcFillList_m.end();
    }

  //============================================================
  // touches operations
  //============================================================

  // Find all subdomains that touch on a given domain, and insert the
  // intersection of these subdomains into the given output iterator.
  // Return the number of touching elements. This version of touches
  // can build either pointers or objects.

  template <class OtherDomain, class OutIter, class ConstructTag>
  int touches(const OtherDomain &fulld, OutIter o,
				 const ConstructTag &ctag) const
    {
      int i;

      // Make sure we have a valid layout

      PAssert(initialized());

      // Figure the type of the domain resulting from the intersection

      typedef typename 
	IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;

      // We only need to do touches for the overlapping domain.  If there is
      // nothing left, we can just return.

      OutDomain_t d = intersect(domain_m, fulld);
      if (d.empty())
	return 0;

      // Create an object of this output domain type for use later.

      OutDomain_t outDomain = Pooma::NoInit();

      // Generate the type of the node pushed on the output iterator.

      typedef Node<OutDomain_t,Domain_t> OutNode_t;
  
      int hiAxisIndex[Dim];
      int loAxisIndex[Dim];
      Loc<Dim> curnode;

      // Find the min and max block indices of the touching domains in each
      // dimension.

      for (i=0;i < Dim; ++i)
	{
	  loAxisIndex[i] =
	    *((map_m[i].touch(Interval<1>(d[i].first(),d[i].first()))).first);
	  PAssert(loAxisIndex[i] >= 0 && 
		  loAxisIndex[i] <= blocks_m[i].first());

	  hiAxisIndex[i] =
	    *((map_m[i].touch(Interval<1>(d[i].last(),d[i].last() ))).first);
	  PAssert(hiAxisIndex[i] >= 0 && 
		  hiAxisIndex[i] <= blocks_m[i].first());

	  // if this is a reversed range, swap indexes. 

	  if(loAxisIndex[i]>hiAxisIndex[i])
	    {
	      int tmp = hiAxisIndex[i];
	      hiAxisIndex[i] = loAxisIndex[i];
	      loAxisIndex[i] = tmp;
	    }


	  curnode[i] = loAxisIndex[i];


	}


      // Go through all the blocks and output the values.

      int count=0;
      while(1)
	{ 
	  // curnode is the current block ... convert it to a serial index

	  int nodeListIndex = blockIndex(curnode);

	  // we can skip this block if it is empty

	  if (!all_m[nodeListIndex]->domain().empty())
	    {
	      // Make sure that block is OK ... this is a sanity check.

	      outDomain = intersect(fulld,all_m[nodeListIndex]->domain());
	      PAssert(!outDomain.empty());

	      // Output this block.

	      *o = touchesConstruct(outDomain,
				    all_m[nodeListIndex]->allocated(),
				    all_m[nodeListIndex]->affinity(),
				    all_m[nodeListIndex]->context(),
				    all_m[nodeListIndex]->globalID(),
				    all_m[nodeListIndex]->localID(),
				    ctag);
	      ++count;
	    }

	  // Move on to the next block

	  curnode[0] += 1;
	  for (i=0; i < Dim; ++i)
	    {
	      if (curnode[i] == (hiAxisIndex[i]+1))
		{
		  if (i==(Dim-1))
		    break;

		  curnode[i] = loAxisIndex[i];
		  curnode[i+1] += 1;
		}
	    }

	  // Are we at the end?

	  if (curnode[Dim-1] == (hiAxisIndex[Dim-1]+1))
	    break;
	}

      return count;
    }


//-----------------------------------------------------------------------------
//
// Find all subdomains that touch on a given domain, and insert the
// intersection of these subdomains into the given output iterator.
// Return the number of touching elements. This version of touches
// can build either pointers or objects.
//
//-----------------------------------------------------------------------------

template <class OtherDomain, class OutIter, class ConstructTag>
int touchesAlloc(const OtherDomain &fulld, OutIter o,
				      const ConstructTag &ctag) const 
{
  int i;

  // Make sure we have a valid layout

  PAssert(initialized());

  // Figure the type of the domain resulting from the intersection.

  typedef typename 
    IntersectReturnType<Domain_t,OtherDomain>::Type_t OutDomain_t;

  // We only need to do touches for the overlapping domain.  If there is
  // nothing left, we can just return.

  OutDomain_t d = intersect(domain_m, fulld);
  if (d.empty())
    return 0;

  // Create an object of this output domain type for use later.

  OutDomain_t outDomain = Pooma::NoInit();

  // Generate the type of the node pushed on the output iterator.

  typedef Node<OutDomain_t,Domain_t> OutNode_t;
  
  int hiAxisIndex[Dim];
  int loAxisIndex[Dim];
  Loc<Dim> curnode;

  // Find the min and max block indices of the touching domains in each
  // dimension.

  for (i=0; i < Dim; ++i)
    {
      loAxisIndex[i] =
	*((mapAloc_m[i].touch(Interval<1>(d[i].first(),d[i].first()))).first);
      PAssert(loAxisIndex[i] >= 0 && 
	      loAxisIndex[i] < blocks_m[i].first());

      hiAxisIndex[i] = 
	*((mapAloc_m[i].touch(Interval<1>(d[i].last(),d[i].last()))).first);
      PAssert(hiAxisIndex[i] >= 0 && 
	      hiAxisIndex[i] < blocks_m[i].first());

      // if this is a reversed range, swap indexes. 

      if(loAxisIndex[i]>hiAxisIndex[i])
	{
	  int tmp = hiAxisIndex[i];
	  hiAxisIndex[i] = loAxisIndex[i];
	  loAxisIndex[i] = tmp;
	}

      curnode[i] = loAxisIndex[i];
    }

  // Go through all the blocks and output the values.

  int count=0;
  while(1)
    {
      // curnode is the current block ... convert it to a serial index

      int nodeListIndex = blockIndex(curnode);

      // we can skip this block if it is empty

      if (!all_m[nodeListIndex]->domain().empty())
	{
	  // Make sure that block is OK ... this is a sanity check.

	  outDomain = intersect(fulld,all_m[nodeListIndex]->allocated());

	  PAssert(!outDomain.empty());

	  // Output this block.

	  *o = touchesConstruct(outDomain,
				all_m[nodeListIndex]->allocated(),
				all_m[nodeListIndex]->affinity(),
				all_m[nodeListIndex]->context(),
				all_m[nodeListIndex]->globalID(),
				all_m[nodeListIndex]->localID(),
				ctag);
	  ++count;
	}
	
      // Move on to the next block

      curnode[0] += 1;
      for (i=0; i < Dim; ++i)
	{
	  if (curnode[i] == (hiAxisIndex[i]+1))
	    {
	      if (i == (Dim-1))
		break;

	      curnode[i] = loAxisIndex[i];
	      curnode[i+1] += 1;
	    }
	}

      // Are we at the end?

      if (curnode[Dim-1] == (hiAxisIndex[Dim-1]+1))
	break;
    }

  return count;
}

   void sync();

  template<class Out>
  void print(Out & ostr)
  {
    int i;
    ostr << " hasInternalGuards_m, hasExternalGuards_m " <<
      hasInternalGuards_m <<" " << hasExternalGuards_m <<std::endl;
    ostr << " internalGuards_m " ;
    for ( i=0;i<Dim;++i) 
      ostr << internalGuards_m.upper(i)<<"-"<<internalGuards_m.lower(i)<<" ";
    ostr <<std::endl;
    ostr << " externalGuards_m " ;
    for ( i=0;i<Dim;++i) 
      ostr << externalGuards_m.upper(i)<<"-"<<externalGuards_m.lower(i)<<" ";
    ostr <<std::endl;
    FillIterator_t gstart = gcFillList_m.begin();
    FillIterator_t gend = gcFillList_m.end();
    ostr<< " gcFillList_m " <<std::endl;
    for( ; gstart!=gend ; ++gstart)
      ostr<<"       "
	  <<gstart->domain_m<<" "
	  <<gstart->ownedID_m<<" "
	  <<gstart->guardID_m<<std::endl;
  }    

private:
  //============================================================
  // Private Methods
  //============================================================

  // This function calculates the cached guard-cell filling information.
  
  void calcGCFillList();

  // This function recalculates what the total domain of each patch
  // should be, since this can change due to dynamic operations.

  void calcDomains();

  // This function recalculates the domain maps, since this can
  // change due to dynamic operations.

  void calcMaps() const;
  void calcAllocMaps() const;

  // compute the serial index of the given block

  inline int blockIndex(const Loc<Dim> &loc) const
    {
      int pos = loc[0].first();
      for (int i=1; i < Dim; ++i)
	pos += loc[i].first() * blockStrides_m[i];
      return pos;
    }

  //============================================================
  // Private Data
  //============================================================

//   // copy of the Grid<Dim> object that can be used to describe
//   // the layout. 
// **** Not currently stored, however it seems to me that we want to
// store this information, as with it, and the info in LayoutBase
// et al, the layout can be easily reconstructed. 
//    Grid<Dim> grid_m;


  // Is the current layout information out-of-date (due to dynamic ops
  // that have yet to have sync called?)

  bool dirtyLayout_m;

  // Strides used to convert block i,j,k indices to serialized indices.

  int blockStrides_m[Dim];

  // This domain map is created for each axis. For each axis, the domain
  // is tiled exactly, with no overlap by each DomanMapNode. 

  mutable DomainMap<Interval<1>,AxisIndex_t> map_m[Dim];

  // This DM is initialized with the real domain, extended at either end by
  // the external guard cell size. Internally, the DMN's overlap by the amount
  // of the internal guard cell layers. 

  // This will only yield ordered touches
  // (where the touch operation returns iterators that deref to an increasing
  // fortran style index) if the domain on which the grid is built is uniformly
  // increasing or decreasing (i.e. Grid(Interval<1>(0,5),Range<1>(8,4,-2)) may
  // not yield ordered results.)

  mutable DomainMap<Interval<1>,AxisIndex_t> mapAloc_m[Dim]; 
};


//-----------------------------------------------------------------------------
// Full Description:
//
// GridLayout is a Layout class that breaks an N-dimensional
// Interval into sub-domains arranged in an N-dimensional
// grid, where the sub-domain sizes are specified by a Grid domain object
//
// This is an alternative to the more general tile Layout class that should
// perform faster since subdomains can be found using a set of 1-dimensional
// domainMap's, rather than by a more general search.
//
// To construct a GridLayout, you can do any of the following:
//
//   1) provide a global domain, and let the GridLayout perform
//      its default partitioning by just using one single block;
//
//   2) provide a global domain, a Loc with the number of blocks to
//      use along each dimension, and an optional context number;
//      
//   3) provide a global domain and a GridPartition  or UniformGridPartition 
//      object.
//        ** The above three methods are the same as for UniformGridLayout
//   4) provide a Grid domain object
//
//   5) provide a Grid domain object and a GuardLayers objects to create
//      the grid with external and/or internal guard cells. 
//      
//-----------------------------------------------------------------------------

template <int Dim>
class GridLayout : public LayoutBase<Dim,GridLayoutData<Dim> >,
                   public Observable<GridLayout<Dim> >,
                   public Observer<GridLayoutData<Dim> >
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================

  // General public typedefs.

  typedef GridLayout<Dim>                      This_t;
  typedef Observable<This_t>                   Observable_t;
  typedef GridLayoutData<Dim>                  LayoutData_t;
  typedef typename LayoutData_t::Domain_t      Domain_t;
  typedef typename LayoutData_t::BaseDomain_t  BaseDomain_t;
  typedef typename LayoutData_t::Context_t     Context_t;
  typedef typename LayoutData_t::ID_t          ID_t;
  typedef typename LayoutData_t::Value_t       Value_t;
  typedef typename LayoutData_t::List_t        List_t;
  typedef DynamicEvents::PatchID_t             PatchID_t;
  typedef DynamicEvents::CreateSize_t          CreateSize_t;
  typedef GuardLayers<Dim>                     GuardLayers_t;

  // Iterator through nodes. Basically the same as the vector iterator
  // except it dereferences automatically.  
  
  typedef DerefIterator<Value_t>               iterator;
  typedef ConstDerefIterator<Value_t>          const_iterator;
   
  // Iterator through guard-cell-fill requests. 
  
  typedef typename LayoutData_t::GCFillInfo_t GCFillInfo_t;
  
  typedef typename 
    std::vector<GCFillInfo_t>::const_iterator FillIterator_t;
  
  // Enumerations.

  enum { dimensions = Dim };
  enum { repartitionEvent = 1 };
  enum { dynamic = true };

  
  //============================================================
  // Constructors
  //============================================================

  // The default constructor does not initialize the layout.  In this
  // case, layout initialization must be completed with the
  // 'initialize' method before the layout can be used.  A default
  // layout has an empty global domain, and empty subdomain lists.
  
  GridLayout();

  // Construct a layout with nothing else but a global domain.  In
  // this case, a default partitioner will be used, the GridPartition
  // object, which will just make a grid with one block and no guard cells.
  
  GridLayout(const Domain_t &,const DistributedTag &);

  GridLayout(const Domain_t &,const ReplicatedTag &);

  // Same, but also specify guard cell info

  GridLayout(const Domain_t &,
             const GuardLayers_t &,const DistributedTag &);

  GridLayout(const Domain_t &,
             const GuardLayers_t &,const ReplicatedTag &);

  // Domain + block count constructors.
  // In this case, the user specifies the global domain and the number
  // of blocks in each dimension, which will cause the domain to be
  // partitioned in a uniform manner. One or two GuardLayers objects may
  // be specified. If only one is specified, it is used for both
  // internal and external guards.
  GridLayout(const Domain_t &,
	     const Loc<Dim> &,const DistributedTag &);

  GridLayout(const Domain_t &,
	     const Loc<Dim> &, 
	     const GuardLayers_t &,const DistributedTag &);

  GridLayout(const Domain_t &,
	     const Loc<Dim> &, 
	     const GuardLayers_t &,
	     const GuardLayers_t &,const DistributedTag &);

  GridLayout(const Domain_t &,
	     const Loc<Dim> &,const ReplicatedTag &);

  GridLayout(const Domain_t &,
	     const Loc<Dim> &, 
	     const GuardLayers_t &,const ReplicatedTag &);

  GridLayout(const Domain_t &,
	     const Loc<Dim> &, 
	     const GuardLayers_t &,
	     const GuardLayers_t &,const ReplicatedTag &);

  // Domain + Grid Domain constructors. 
  // Like the block count constructors, but a Grid object is specified,
  // allowing a general non-uniform grid to be created.  Again, one or two
  // GuardLayers objects may be specified. If only one is specified, it is
  // used for both internal and external guards.
  
  GridLayout(const Grid<Dim> &,
	     const DistributedTag &);

  // Internal guard layers same as external guard layers, specified by gcs:
  GridLayout(const Grid<Dim> &,
	     const GuardLayers_t &,
	     const DistributedTag &);

  // Independently-specified internal and external guard layers:
  GridLayout(const Grid<Dim> &,
	     const GuardLayers_t &,
	     const GuardLayers_t &,
	     const DistributedTag &);

  GridLayout(const Grid<Dim> &,
	     const ReplicatedTag &);

  // Internal guard layers same as external guard layers, specified by gcs:
  GridLayout(const Grid<Dim> &,
	     const GuardLayers_t &,
	     const ReplicatedTag &);

  // Independently-specified internal and external guard layers:
  GridLayout(const Grid<Dim> &,
	     const GuardLayers_t &,
	     const GuardLayers_t &,
	     const ReplicatedTag &);

  template <class Partitioner>
  GridLayout(const Domain_t &,
	     const Partitioner &,
	     const DistributedTag &);

  template <class Partitioner>
  GridLayout(const Domain_t &,
	     const Partitioner &,
	     const ReplicatedTag &);
  
  // Domain + partition + mapper constructor. 

  template <class Partitioner>
  GridLayout(const Domain_t &,
	     const Partitioner &,
	     const ContextMapper<Dim> &);

  // Copy constructor & assignment operator
  // Shallow copies with reference counting.

  GridLayout(const This_t &);

  This_t &operator=(const This_t &);
  

  //============================================================
  // Destructor
  //============================================================

  // The actual data will be cleaned up by the LayoutData_t destructor
  // if all references to the data go away.
  // If any Observers remain, they will be notified by the Observable 
  // destructor. 
  
  inline ~GridLayout() 
    { 
      pdata_m->detach(*this);
    }


  //============================================================
  // Initialize methods
  //============================================================

  // Initialize a layout with nothing else but a global domain.  In
  // this case, a default partitioner will be used, the GridPartition
  // object, which will try to make a grid with one block.
  
  void initialize(const Domain_t &,
		  const DistributedTag &);

  void initialize(const Domain_t &,
		  const GuardLayers_t &,
		  const DistributedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const DistributedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const GuardLayers_t &,
		  const DistributedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &, 
		  const GuardLayers_t &,
		  const GuardLayers_t &,
		  const DistributedTag &);

  void initialize(const Grid<Dim> &,
		  const DistributedTag &);

  void initialize(const Grid<Dim> &,
		  const GuardLayers_t &,
		  const DistributedTag &);

  void initialize(const Grid<Dim> &,
		  const GuardLayers_t &,
		  const GuardLayers_t &,
		  const DistributedTag &);

  template <class Partitioner>
  void initialize(const Domain_t &,
		  const Partitioner &,
		  const DistributedTag &);

  // ReplicatedTag
  void initialize(const Domain_t &,
		  const ReplicatedTag &);

  void initialize(const Domain_t &,
		  const GuardLayers_t &,
		  const ReplicatedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const ReplicatedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &,
		  const GuardLayers_t &,
		  const ReplicatedTag &);

  void initialize(const Domain_t &, 
		  const Loc<Dim> &, 
		  const GuardLayers_t &,
		  const GuardLayers_t &,
		  const ReplicatedTag &);

  void initialize(const Grid<Dim> &,
		  const ReplicatedTag &);

  void initialize(const Grid<Dim> &,
		  const GuardLayers_t &,
		  const ReplicatedTag &);

  void initialize(const Grid<Dim> &,
		  const GuardLayers_t &,
		  const GuardLayers_t &,
		  const ReplicatedTag &);

  template <class Partitioner>
  void initialize(const Domain_t &,
		  const Partitioner &,
		  const ReplicatedTag &);

  // Domain + partition + mapper initializer.

  template <class Partitioner>
  void initialize(const Domain_t &,
		  const Partitioner &,
		  const ContextMapper<Dim> &);

  // Used by the I/O or data management system to initialize the layout based
  // on detailed state information previously stored. Since this is specialized
  // for the I/O system, no trailing tag is used. 

  void initialize(const Domain_t& idom,
		  const List_t& nodes,
		  const Loc<Dim>& blocks,
		  bool hasIG, bool hasEG,
		  const GuardLayers_t& ig,
		  const GuardLayers_t& eg);

  //============================================================
  // GridLayout-specific accessors
  //============================================================

  // Return the number of blocks in each dimension, as a Loc.
  
  inline const Loc<Dim> &blocks() const 
    {
      return pdata_m->blocks();
    }

 
  //============================================================
  // Repartition the layout using a new Partitioner scheme.  The
  // initial domain lists are cleared out, the partitioner is invoked,
  // and then all the observers are notified.  This can only be done
  // with a partitioner that generates grid-like blocks.

  template <class Partitioner>
  inline bool repartition(const Partitioner &gp,const ContextMapper<Dim> &cm)
    { 
      pdata_m->initialize(domain(),gp,cm);
      // Can't this just be Observable_t::notify(repartitionEvent)???
      pdata_m->notify(repartitionEvent); 
    }

  void sync()
    { 
      pdata_m->sync();
    }


  //============================================================
  // Observer methods
  //============================================================

  // Respond to events generated by the LayoutData_t.
  // These are just passed on to our observers.

  virtual void notify(LayoutData_t &d, const ObserverEvent &event)
    {
      // We should only get this message from our LayoutData_t object

      PAssert(&d == pdata_m.rawPointer());
      Observable_t::notify(event);
    }


  //============================================================
  // Output
  //============================================================

  // Print a GridLayout on an output stream

  template <class Ostream>
  void print(Ostream &ostr) const {
    ostr << "GridLayout " << ID() << " on global domain " 
      << domain() << ":" << '\n';
    ostr << "   Total subdomains: " << sizeGlobal() << '\n';
    ostr << "   Local subdomains: " << sizeLocal() << '\n';
    ostr << "  Remote subdomains: " << sizeRemote() << '\n';
    ostr << "        Grid blocks: " << blocks() << '\n';
    typename GridLayout<Dim>::const_iterator a;
    for (a = beginGlobal(); a != endGlobal(); ++a)
      ostr << "  Global subdomain = " << *a << '\n';
    for (a = beginLocal(); a != endLocal(); ++a)
      ostr << "   Local subdomain = " << *a << '\n';
    for (a = beginRemote(); a != endRemote(); ++a)
      ostr << "  Remote subdomain = " << *a << '\n';
    pdata_m->print(ostr);
  }

#if !POOMA_NO_TEMPLATE_FRIENDS

  //private:

  template <int Dim1, int Dim2>
  friend class GridLayoutView;

#endif

  //============================================================
  // Data
  //============================================================

  // GridLayout stores its data in a RefCounted class to
  // simplify memory management.
  
  friend class GridLayoutData<Dim>;

};


//-----------------------------------------------------------------------------
//
// Full Description:
// GridLayoutViewData<Dim, Dim2>
//
// The data object held by a GridLayoutView object.
//
//-----------------------------------------------------------------------------


template <int Dim, int Dim2>
class GridLayoutViewData : 
                public LayoutBaseViewData<Dim, Dim2, GridLayout<Dim2> >,
                public RefCounted
{
public:

  typedef GridLayout<Dim2>                  Layout_t;
  typedef GridLayoutView<Dim, Dim2>         ViewLayout_t;

  typedef Interval<Dim>                     Domain_t;
  typedef Range<Dim2>                       BaseDomain_t;
  typedef int                               Context_t;
  typedef Unique::Value_t                   ID_t;

  typedef typename Layout_t::Domain_t       AllocatedDomain_t;
  typedef ViewIndexer<Dim,Dim2>             Indexer_t;

  typedef Node<Domain_t,AllocatedDomain_t>  Value_t;
  typedef std::vector<Value_t *>            List_t;        // for convenience
  typedef GuardLayers<Dim>                  GuardLayers_t; // for convenience

  typedef GridLayoutViewData<Dim,Dim2>      LayoutData_t;
  
  // Enumerations.

  enum { dim = Dim };
  enum { dim2 = Dim2};
  

  //============================================================
  // Constructors
  //============================================================

  GridLayoutViewData() { }
  
  template <class DT>
  inline GridLayoutViewData(const Layout_t &layout, const Domain<Dim, DT> &dom)
  : LayoutBaseViewData<Dim,Dim2,GridLayout<Dim2> >(layout,dom)
    // layout_m(layout), indexer_m(dom), 
    //id_m(Unique::get()), subdomainsComputed_m(false),
    //internalGuards_m(layout.internalGuards()),
    //externalGuards_m(layout.externalGuards())
  { 
  }

  template <class DT>
  inline GridLayoutViewData(const Layout_t &layout, 
                            const SliceDomain<DT> &dom)
  :LayoutBaseViewData<Dim,Dim2,GridLayout<Dim2> >(layout,dom)
  { 
  }

  template <class DT>
  GridLayoutViewData(const ViewLayout_t &layout, 
                     const Domain<Dim, DT> &dom)
  : LayoutBaseViewData<Dim,Dim2,GridLayout<Dim2> >(
					      layout.pdata_m->layout_m,
					      layout,
					      layout.pdata_m->indexer_m, 
					      dom,
					      layout.internalGuards(),
					      layout.externalGuards())
  {
  }

  template <int OrigDim, class DT>
  GridLayoutViewData(const GridLayoutView<OrigDim, Dim2> &layout, 
                     const SliceDomain<DT> &dom)
    : LayoutBaseViewData<Dim,Dim2,GridLayout<Dim2> >(
			     layout.pdata_m->layout_m,
			     layout,
			     Indexer_t(layout.pdata_m->indexer_m,dom),
			     dom)
  { 
  }

  // Destructor

  ~GridLayoutViewData() 
  {
    typename List_t::iterator a;
    for (a = all_m.begin(); a != all_m.end(); ++a)
      delete (*a);
  }

};

//-----------------------------------------------------------------------------
//
// Full Description:
// GridLayoutView<Dim, Dim2>
//
// GridLayoutView is a Layout class that provides a view of an
// existing GridLayout object. Dim is the logical dimension of
// the layout. Dim2 is the dimension of the GridLayout
// contained within.
//
// To construct a GridLayoutView, you need an existing
// GridLayout or a GridLayoutView and the subdomain that
// is being viewed. This class does not have a useful default
// constructor since it is based on an existing GridLayout.
//
// Once created, GridLayoutView has the same interface as
// Layout (see Layout.h). It also provides this extra interface:
//
// int globalID(const Loc<Dim> &pos) : return the globalID
//    of the node that contains the point.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
class GridLayoutView  
: public LayoutBaseView<Dim, Dim2, GridLayoutViewData<Dim,Dim2> >
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  // Enumerations.
  enum { dimensions = Dim };
  enum { dim = Dim };
  enum { dim2 = Dim2};

  // General public typedefs.

  typedef GridLayoutViewData<Dim, Dim2>            LayoutData_t; 

  typedef typename LayoutData_t::Domain_t          Domain_t;
  typedef typename LayoutData_t::BaseDomain_t      BaseDomain_t;
  typedef typename LayoutData_t::Context_t         Context_t;
  typedef typename LayoutData_t::ID_t              ID_t;
  
  typedef typename LayoutData_t::Layout_t          Layout_t;
  typedef typename LayoutData_t::AllocatedDomain_t AllocatedDomain_t;
  typedef typename LayoutData_t::Value_t           Value_t;
  
  typedef typename LayoutData_t::List_t            List_t;
  typedef typename LayoutData_t::Indexer_t         Indexer_t;
  typedef typename LayoutData_t::GuardLayers_t     GuardLayers_t;

  typedef GridLayoutView<Dim, Dim2>                This_t;       // convenience
  typedef GridLayoutView<Dim, Dim2>                ViewLayout_t; // convenience
  typedef LayoutBaseView<Dim, Dim2, LayoutData_t>  Base_t;
  
  // Iterator through nodes. Basically the same as the vector iterator
  // except it dereferences automatically.
  
  typedef DerefIterator<Value_t>                   iterator;
  typedef ConstDerefIterator<Value_t>              const_iterator;


  //============================================================
  // Constructors
  //============================================================

  // Default constructor.
  
  GridLayoutView()
    : Base_t(new LayoutData_t())
  { }
  
  // Constructor building a GridLayoutView from a
  // GridLayout and a non-slice domain like an Interval<Dim> or
  // Range<Dim>.
  
  template <class DT>
  GridLayoutView(const Layout_t &layout, const Domain<Dim2, DT> &dom)
    : LayoutBaseView<Dim,Dim2,GridLayoutViewData<Dim,Dim2> >
  (new GridLayoutViewData<Dim,Dim2>(layout,dom)) 
  { }

  // Constructor building a GridLayoutView from a
  // GridLayout and a slice domain like a
  // SliceInterval<Dim2,Dim> or SliceRange<Dim2,Dim>.
  
  template <class DT>
  GridLayoutView(const Layout_t &layout, const SliceDomain<DT> &dom)
    : LayoutBaseView<Dim,Dim2,GridLayoutViewData<Dim,Dim2> >
  (new GridLayoutViewData<Dim,Dim2>(layout,dom))  
  { }
  
  // Constructor building a GridLayoutView from another
  // GridLayoutView and a non-slice domain like an
  // Interval<Dim> or Range<Dim>.
  
  template <class DT>
  GridLayoutView(const ViewLayout_t &layout, const Domain<Dim, DT> &dom)
 : LayoutBaseView<Dim,Dim2,GridLayoutViewData<Dim,Dim2> >
  (new GridLayoutViewData<Dim,Dim2>(layout,dom))
  { }

  // Constructor building a GridLayoutView from another
  // GridLayoutView and a slice domain like a
  // SliceInterval<Dim2,Dim> or SliceRange<Dim2,Dim>.
  
  template <int OldViewDim, class DT>
  GridLayoutView(const GridLayoutView<OldViewDim, Dim2> &layout, 
                        const SliceDomain<DT> &dom)
  : LayoutBaseView<Dim,Dim2,GridLayoutViewData<Dim,Dim2> > 
  (new GridLayoutViewData<Dim,Dim2>(layout,dom)) 
  { }

  // Copy constructor & assignment operator
  // Shallow copies with reference counting.
  
  inline GridLayoutView(const This_t &model) 
    : LayoutBaseView<Dim,Dim2,GridLayoutViewData<Dim,Dim2> >(model.pdata_m)
  { }
  
  inline This_t &operator=(const This_t &model) 
  {
    if (this != &model)
      {
        pdata_m = model.pdata_m;
      }
    return *this;
  }


  //============================================================
  // Destructor
  //============================================================

  // The actual data will be cleaned up by the GridLayoutData
  // destructor if all references to the data go away, so there is
  // nothing to do here.
   
  inline ~GridLayoutView() 
  { }

  //============================================================
  // Output
  //============================================================
    
  // Print a GridLayoutView on an output stream

  template <class Ostream>
  void print(Ostream &ostr) const 
  {
    ostr << "GridLayoutView " << ID() << " on global domain " 
      << domain() << ":" << '\n';
    ostr << "   Base ID:          " << baseID() << '\n';
    ostr << "   Base domain:      " << baseDomain() << '\n';
    ostr << "   Total subdomains: " << sizeGlobal() << '\n';
    ostr << "   Local subdomains: " << sizeLocal() << '\n';
    ostr << "  Remote subdomains: " << sizeRemote() << '\n';
    const_iterator a;
    for (a = beginGlobal(); a != endGlobal(); ++a)
      ostr << "  Global subdomain = " << *a << '\n';
    for (a = beginLocal(); a != endLocal(); ++a)
      ostr << "   Local subdomain = " << *a << '\n';
    for (a = beginRemote(); a != endRemote(); ++a)
      ostr << "  Remote subdomain = " << *a << '\n';
  }

#if !POOMA_NO_TEMPLATE_FRIENDS

  //private:

  template <int OtherDim, int OtherDim2>
  friend class GridLayoutView;

  template <int OtherDim, int OtherDim2>
  friend class GridLayoutViewData;

#endif

  //============================================================
  // Private utility functions
  //============================================================

  // Fill our subdomain lists.
  
  void computeSubdomains() const { pdata_m->computeSubdomains(); }
   
  //============================================================
  // Data
  //============================================================

  // The data is stored in a RefCounted class to simplify memory
  // management.  This is probably not as important for ViewLayout
  // classes as for Layout classes, but we do it for
  // consistency. Currently ViewLayouts are not observed directly by
  // anyone. Of course, the Layout that we have a copy of is observed.

};
//============================================================
// NewDomain1 traits classes for GridLayout and GridLayoutView
//============================================================

//-----------------------------------------------------------------------------
//
// This is so an array can be initialized with a GridLayout.
//
//-----------------------------------------------------------------------------

template <int Dim>
struct NewDomain1<GridLayout<Dim> >
{
  typedef GridLayout<Dim> &Type_t;

  inline static Type_t combine(const GridLayout<Dim> &a)
    {
      return const_cast<Type_t>(a);
    }
};

//-----------------------------------------------------------------------------
//
// This is so an array can be initialized with a GridLayoutView.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
struct NewDomain1<GridLayoutView<Dim, Dim2> >
{
  typedef GridLayoutView<Dim, Dim2> &Type_t;

  inline static Type_t combine(const GridLayoutView<Dim, Dim2> &a)
    {
      return const_cast<Type_t>(a);
    }
};

//-----------------------------------------------------------------------------
//
// ostream inserters for GridLayout and GridLayoutView:
//
//-----------------------------------------------------------------------------

template <int Dim>
std::ostream &operator<<(std::ostream &ostr, 
			 const GridLayout<Dim> &layout)
{
  layout.print(ostr);
  return ostr;
}

template <int Dim, int Dim2>
std::ostream &operator<<(std::ostream &ostr, 
			 const GridLayoutView<Dim, Dim2> &layout)
{
  layout.print(ostr);
  return ostr;
}


// } // namespace POOMA

#include "Layout/GridLayout.cpp"

#endif // POOMA_LAYOUT_GRIDLAYOUT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: GridLayout.h,v $   $Author: jac $
// $Revision: 1.107 $   $Date: 2000/06/27 01:58:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
