//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Array/Array.h"
#include "IO/ObjectSets.h"

//-----------------------------------------------------------------------------
// Test program for object sets. Part 1 writes the file, part 2 reads or
// recovers.
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  Pooma::Tester tester(argc, argv);

  tester.out() <<"Test Object Sets"<< std::endl;

  // Create an new object storage set.
  ObjectStorageSet obset("myData.dat",Create);
	
  // Create a set of simple and complex objects to store.
  int k=200;
  double d=3.1415;

  Interval<3> I3(6,6,6);
  Array<3> a0(I3);
  Array<3, double, CompressibleBrick> a1(I3);
  
  Loc<3> blocks(1,1,2);
  UniformGridPartition<3> partition(blocks);   
  UniformGridLayout<3> layout(I3, partition, ReplicatedTag());
  
  Array<3, double, MultiPatch<UniformTag,Brick> > 
    a2(layout);

  a0 = 0.0;
  a1 = 1.0;
  a2 = 2.0;

  tester.out()<<"k="<<k<<std::endl;
  tester.out()<<"d="<<d<<std::endl;
  tester.out()<<"a0="<<a0<<std::endl;
  tester.out()<<"a1="<<a1<<std::endl;
  tester.out()<<"a2="<<a2<<std::endl;

  ObjectID_t kid= obset.store(k,"k");
  tester.out()<<"int object stored, ID="<<kid<<std::endl;
  ObjectID_t did= obset.store(d,"d");
  tester.out()<<"double object stored, ID="<<did<<std::endl;
  ObjectID_t a0id= obset.store(a0,"a0");
  tester.out()<<"Array<3> object stored, ID="<<a0id<<std::endl;
  ObjectID_t a1id= obset.store(a1,"a1");
  tester.out()<<"Array<3, double, CompressibleBrick> > object stored, ID="<<a1id<<std::endl;
  ObjectID_t a2id= obset.store(a2,"a2");
  tester.out()<<"Array<3, double, MultiPatch<UniformTag,Brick> > object stored, ID="
	   <<a2id<<std::endl;
  
  // Ask for the number of objects in the set.
  int numobj= obset.numObjects();
  tester.out()<<numobj<<" objects in the object set"<<std::endl;
  tester.check(numobj==5);
  
  // Store multiple instances of "mydouble".
  d= 1.234;
  tester.out()<<"d="<<d<<std::endl;
  obset.store(d,"d");
  d= 5.678;
  tester.out()<<"d="<<d<<std::endl;
  obset.store(d,"d");
  
  // Ask for the number of objects in the set.
  numobj= obset.numObjects();
  tester.out()<<numobj<<" objects in the object set"<<std::endl;
  tester.check(numobj==7);
  
  // Close.
  obset.close();

  // Restore the data set written in part 1 as a separate object.
  ObjectStorageSet myset("myData.dat", Restore);
	
  // Ask for the number of objects in the set and check.
  numobj= myset.numObjects();
  tester.out()<<numobj<<" objects in the object set"<<std::endl;
  tester.check(numobj==7);

  // Ask for the types of the objects and print them.
  std::string typeName;
  ObjectID_t oid;
  for(int i=0;i<numobj;i++){
    oid= i+1;
    typeName= myset.objectType(oid);
    tester.out()<<"Object "<<oid<<" is type "<<typeName<<std::endl;
  }

  // Create appropriate target objects for retrieval.
  int k2;
  double d2;
  Array<3> a02;
  Array<3, double, CompressibleBrick> a12;
  Array<3, double, MultiPatch<UniformTag,Brick> > a22;

  // Retrieve the objects.

  kid= myset.retrieve(k2,"k");
  tester.out()<<"int object retrieved, ID="<<kid<<std::endl;
  did= myset.retrieve(d2,"d");
  tester.out()<<"double object retrieved, ID="<<did<<std::endl;
  a0id= myset.retrieve(a02,"a0");
  tester.out()<<"Array<3> object retrieved, ID="<<a0id<<std::endl;
  a1id= myset.retrieve(a12,"a1");
  tester.out()<<"Array<3, double, CompressibleBrick> > object retrieved, ID="<<a1id<<std::endl;
  a2id= myset.retrieve(a22,"a2");
  tester.out()<<"Array<3, double, MultiPatch<UniformTag,Brick> > object retrieved, ID="
	   <<a2id<<std::endl;
  
  // Print out the values.

  tester.out()<<"k2="<<k2<<std::endl;
  tester.out()<<"d2="<<d2<<std::endl;
  tester.out()<<"a02="<<a02<<std::endl;
  tester.out()<<"a12="<<a12<<std::endl;
  tester.out()<<"a22="<<a22<<std::endl;

  // Check the results.
  bool result;
  result= k2 == 200;
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= d2 == 3.1415;
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(a02 == 0.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(a12 == 1.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(a22 == 2.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);

  // Ask for the number of objects called "d".
  int ndouble= myset.count("d");
  tester.out()<<"Number of 'd' instances is "<<ndouble<<std::endl;
  tester.check(ndouble==3);
  
  // If the count is correct, get a set of object IDs corresponding
  // to the label.
  if(ndouble==3){
    int inst=0;
    double dvals[3]={3.1415, 1.234, 5.678};
    ObjectSubset dset= myset.selectLabel("d");
    ObjectSubset::iterator iter;
    for(iter= dset.begin(); iter!= dset.end(); iter++){
      myset.retrieve(d2,*iter);
      tester.out()<<"d="<<d2<<std::endl;
      tester.check(d2==dvals[inst]);
      inst++;
    }
  }
  
  
  // Close and quit.
  myset.close();
  int retval = 
    tester.results("Test of ObjectStorageSet");
  Pooma::finalize();
  return retval;
}


