//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Utilities/Tester.h"
#include "Pooma/Pooma.h"
#include "IO/Serializers.h"

//-----------------------------------------------------------------------------
// Test of the base serializer capability for I/O
//-----------------------------------------------------------------------------

void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m);
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m){
  tstr.check(cond);
  if(cond){
    tstr.out() << "PASSED "<<m<<std::endl;
  }
  else{
    tstr.out() << "FAILED "<<m<<std::endl;
  }
}

//-----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  // Serialize int, long, float, double, complex, and string values, singly and as
  // arrays to a file, and to a character buffer.

  tester.out() <<"Test of base serializers:"<< std::endl;

  // Open a file for the test.
  std::fstream datafile("baseSerializerTest.dat",
                        std::ios::out | std::ios::binary);

  // Create a buffer for the test.
  char databuffer[1000];

  const int nvals= 5;

  int ints1[nvals]= {1,2,3,4,5};
  long longs1[nvals]= {10,20,30,40,50};
  float floats1[nvals]= {1.1,2.2,3.3,4.4,5.5};
  double doubles1[nvals]= {1.11,2.22,3.33,4.44,5.55};
  std::string strings1[nvals]= {"one","two","three",
				"four","five"};

  // Serialize individual values to the data file
  // and buffer respectively.

  char* databuf= (char*) &databuffer[0];
  for(int i=0;i<nvals;i++){
    // To the data file:
    serialize(datafile, ints1[i]);
    serialize(datafile, longs1[i]);
    serialize(datafile, floats1[i]);
    serialize(datafile, doubles1[i]);
    serialize(datafile, strings1[i]);
    // To data buffer:
    serialize(databuf, ints1[i]);
    serialize(databuf, longs1[i]);
    serialize(databuf, floats1[i]);
    serialize(databuf, doubles1[i]);
    serialize(databuf, strings1[i]);
  }

  // Serialize as arrays to the data file:
  serializeN(datafile, ints1, nvals);
  serializeN(datafile, longs1, nvals);
  serializeN(datafile, floats1, nvals);
  serializeN(datafile, doubles1, nvals);
  serializeN(datafile, strings1, nvals);
  // To the data buffer:
  serializeN(databuf, ints1, nvals);
  serializeN(databuf, longs1, nvals);
  serializeN(databuf, floats1, nvals);
  serializeN(databuf, doubles1, nvals);
  serializeN(databuf, strings1, nvals);

  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile("baseSerializerTest.dat",
                     std::ios::in | std::ios::binary);

  // Reset the buffer pointer.
  databuf= (char*) &databuffer[0];

  // Recover the data.

  int ints2[nvals];
  long longs2[nvals];
  float floats2[nvals];
  double doubles2[nvals];
  std::string strings2[nvals];

  int ints3[nvals];
  long longs3[nvals];
  float floats3[nvals];
  double doubles3[nvals];
  std::string strings3[nvals];

  int ints4[nvals];
  long longs4[nvals];
  float floats4[nvals];
  double doubles4[nvals];
  std::string strings4[nvals];

  int ints5[nvals];
  long longs5[nvals];
  float floats5[nvals];
  double doubles5[nvals];
  std::string strings5[nvals];

  // Deserialize individual values from the data file
  // and from the buffer respectively.

  for(int i=0;i<nvals;i++){
    // From the data file:
    deserialize(ints2[i], dfile);
    deserialize(longs2[i], dfile);
    deserialize(floats2[i], dfile);
    deserialize(doubles2[i], dfile);
    deserialize(strings2[i], dfile);
    // From the data buffer:
    deserialize(ints3[i], databuf);
    deserialize(longs3[i], databuf);
    deserialize(floats3[i], databuf);
    deserialize(doubles3[i], databuf);
    deserialize(strings3[i], databuf);
  }

  // Deserialize as arrays from the data file:
  deserializeN(ints4, dfile, nvals);
  deserializeN(longs4, dfile, nvals);
  deserializeN(floats4, dfile, nvals);
  deserializeN(doubles4, dfile, nvals);
  deserializeN(strings4, dfile, nvals);
  // From the data buffer:
  deserializeN(ints5, databuf, nvals);
  deserializeN(longs5, databuf, nvals);
  deserializeN(floats5, databuf, nvals);
  deserializeN(doubles5, databuf, nvals);
  deserializeN(strings5, databuf, nvals);

  // Check each of the results.
  const int ntypes=5;
  bool result[ntypes]= {true,true,true,true,true};
  tester.out() <<"Checking file storage of individual type instances"<<std::endl;
  for(int i=0;i<nvals;i++){
    result[0]= result[0]&&(ints1[i]==ints2[i]);
    result[1]= result[1]&&(longs1[i]==longs2[i]);
    result[2]= result[2]&&(floats1[i]==floats2[i]);
    result[3]= result[3]&&(doubles1[i]==doubles2[i]);
    result[4]= result[4]&&(strings1[i]==strings2[i]);
  }

  itemCheck(tester,result[0],"int");
  itemCheck(tester,result[1],"long");
  itemCheck(tester,result[2],"float");
  itemCheck(tester,result[3],"double");
  itemCheck(tester,result[4],"string");

  // Reset item flags.
  for(int i=0;i<ntypes;i++){
    result[0]=true;
  }

  tester.out() <<"Checking buffer storage of individual type instances"<<std::endl;
  for(int i=0;i<nvals;i++){
    result[0]= result[0]&&(ints1[i]==ints3[i]);
    result[1]= result[1]&&(longs1[i]==longs3[i]);
    result[2]= result[2]&&(floats1[i]==floats3[i]);
    result[3]= result[3]&&(doubles1[i]==doubles3[i]);
    result[4]= result[4]&&(strings1[i]==strings3[i]);
  }

  itemCheck(tester,result[0],"int");
  itemCheck(tester,result[1],"long");
  itemCheck(tester,result[2],"float");
  itemCheck(tester,result[3],"double");
  itemCheck(tester,result[4],"string");

  // Reset item flags.
  for(int i=0;i<ntypes;i++){
    result[0]=true;
  }

  tester.out() <<"Checking file storage of arrays"<<std::endl;
  for(int i=0;i<nvals;i++){
    result[0]= result[0]&&(ints1[i]==ints4[i]);
    result[1]= result[1]&&(longs1[i]==longs4[i]);
    result[2]= result[2]&&(floats1[i]==floats4[i]);
    result[3]= result[3]&&(doubles1[i]==doubles4[i]);
    result[4]= result[4]&&(strings1[i]==strings4[i]);
  }

  itemCheck(tester,result[0],"int");
  itemCheck(tester,result[1],"long");
  itemCheck(tester,result[2],"float");
  itemCheck(tester,result[3],"double");
  itemCheck(tester,result[4],"string");

  // Reset item flags.
  for(int i=0;i<ntypes;i++){
    result[0]=true;
  }

  tester.out() <<"Checking buffer storage of arrays"<<std::endl;
  for(int i=0;i<nvals;i++){
    result[0]= result[0]&&(ints1[i]==ints5[i]);
    result[1]= result[1]&&(longs1[i]==longs5[i]);
    result[2]= result[2]&&(floats1[i]==floats5[i]);
    result[3]= result[3]&&(doubles1[i]==doubles5[i]);
    result[4]= result[4]&&(strings1[i]==strings5[i]);
  }

  itemCheck(tester,result[0],"int");
  itemCheck(tester,result[1],"long");
  itemCheck(tester,result[2],"float");
  itemCheck(tester,result[3],"double");
  itemCheck(tester,result[4],"string");

  int retval = tester.results("Test of base serializers");

  dfile.close();

  Pooma::finalize();
  return retval;
}


