// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support Pooma Engines
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_ENGINE_IO_H
#define POOMA_IO_ENGINE_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "Pooma/Arrays.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include "IO/DomainIO.h"
#include "IO/LayoutIO.h"
#include <string>

//-----------------------------------------------------------
// CTTI functions for engine-related tag  classes
//-----------------------------------------------------------

inline std::string PoomaCTTI(const Brick&);
inline std::string PoomaCTTI(const Brick&){
  return std::string("Brick");
}

inline std::string PoomaCTTI(const CompressibleBrick&);
inline std::string PoomaCTTI(const CompressibleBrick&){
  return std::string("CompressibleBrick");
}

inline std::string PoomaCTTI(const UniformTag&);
inline std::string PoomaCTTI(const UniformTag&){
  return std::string("UniformTag");
}

inline std::string PoomaCTTI(const GridTag&);
inline std::string PoomaCTTI(const GridTag&){
  return std::string("GridTag");
}

template <class LayoutTag, class PatchTag>
std::string PoomaCTTI(const MultiPatch<LayoutTag,PatchTag>&);
template <class LayoutTag, class PatchTag>
std::string PoomaCTTI(const MultiPatch<LayoutTag,PatchTag>&){
  std::string strng= "MultiPatch<"+ PoomaCTTI(LayoutTag())
    + ","+ PoomaCTTI(PatchTag())+ "> ";
  return strng;
}

//-----------------------------------------------------------
// CTTI function for Engine<Dim,T,EngineTag>
//-----------------------------------------------------------

template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Engine<Dim, T, EngineTag>&);
template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Engine<Dim, T, EngineTag>&){
  char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "Engine<"+
    std::string(numbuf)+ PoomaCTTI(T())
    + ","+ PoomaCTTI(EngineTag())+ ">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for Engine<Dim,T,Brick>
//-----------------------------------------------------------

template <class Stream, int Dim, class T>
int serialize(Stream& s, const Engine<Dim, T, Brick>& engine);
template <class Stream, int Dim, class T>
int serialize(Stream& s, const Engine<Dim, T, Brick>& engine){

  int nBytes=0;

  // Obtain the attributes of the engine.

  // Get the Interval domain and serialize.
  Interval<Dim> dom= engine.domain();
  nBytes+= serialize(s,dom);

  // Get the total brick size.
  int totalSize= dom.size();

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Write these descriptive attributes.
  nBytes+= serialize(s,totalSize);
  nBytes+= serialize(s,elemType);

  // Get a pointer and write the data and serialize.
  T* pdata= &engine(dom.firsts());
  nBytes+= serializeN(s,pdata,totalSize);

  return nBytes;
}

template <class Stream, int Dim, class T>
int deserialize(Engine<Dim, T, Brick>& engine, Stream& s);
template <class Stream, int Dim, class T>
int deserialize(Engine<Dim, T, Brick>& engine, Stream& s){

  typedef Engine<Dim,T,Brick> Engine_t;
  int nBytes=0;

  // Deserialize the domain interval.
  Interval<Dim> dom;
  nBytes+= deserialize(dom,s);

  // Deserialize the descriptive attributes.
  int totalSize;
  std::string elemType;
  nBytes+= deserialize(totalSize,s);
  nBytes+= deserialize(elemType,s);

  // Insist the element type match stored type.
  std::string thisElemType= PoomaCTTI(T());

  PInsist((elemType==thisElemType),
  "Error: source and target engine element types do not match.");

  // Initialize the engine with the restored interval domain
  engine= Engine_t(dom);

  // Insist that the interval size and stored size attribute match.
  int intervalSize= dom.size();
  PInsist((totalSize==intervalSize),
  "Error: source data size and target interval sizes do not match.");

  // Get a data pointer and read the data.
  T* pdata= &engine(dom.firsts());
  nBytes+= deserializeN(pdata,s,totalSize);

  return nBytes;
}

template <int Dim, class T>
int serialSizeof(const Engine<Dim, T, Brick>& engine);
template <int Dim, class T>
int serialSizeof(const Engine<Dim, T, Brick>& engine){

  int nBytes=0;

  // Obtain the attributes of the engine.

  // Get the Interval domain serial size.
  Interval<Dim> dom= engine.domain();
  nBytes+= serialSizeof(dom);

  // Get the total brick size.
  int totalSize= dom.size();

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Get the size of the descriptive attributes.
  nBytes+= sizeof(int);
  nBytes+= serialSizeof(elemType);

  // Safest size computation obtains the size of
  // each element.
  T* pdata= &engine(dom.firsts());
  for(int i=0;i<totalSize;i++){
    nBytes+= serialSizeof(pdata[i]);
  }
  return nBytes;
}

//-----------------------------------------------------------
// Serializers for Engine<Dim,T,CompressibleBrick>
//-----------------------------------------------------------

template <class Stream, int Dim, class T>
int serialize(Stream& s, const Engine<Dim, T, CompressibleBrick>& engine);
template <class Stream, int Dim, class T>
int serialize(Stream& s, const Engine<Dim, T, CompressibleBrick>& engine){

  int nBytes=0;

  // Obtain the attributes of the engine.

  // Get the Interval domain and serialize.
  Interval<Dim> dom= engine.domain();
  nBytes+= serialize(s,dom);

  // Get the total brick size.
  int totalSize= dom.size();

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Write the element type attribute.
  nBytes+= serialize(s,elemType);

  // Check the compression flag and set the data
  // parameters accordingly.
  int compressed;
  int dataSize;
  T* pdata;
  if(engine.compressed()){
    compressed= 1;
    dataSize= 1;
    // Get a pointer to the compressed value.
    pdata= &engine.compressedReadWrite();
  }
  else{
    compressed= 0;
    dataSize= totalSize;
    // Get the uncompressed data block.
    CompressibleBlock<T> cblock= engine.cblock();
    // Get a pointer to the data.
    pdata= cblock.data();
  }

  // Write the compression flag, the data size,
  // and the data.
  nBytes+= serialize(s,compressed);
  nBytes+= serialize(s,dataSize);
  nBytes+= serializeN(s,pdata,dataSize);

  return nBytes;
}

template <class Stream, int Dim, class T>
int deserialize(Engine<Dim, T, CompressibleBrick>& engine, Stream& s);
template <class Stream, int Dim, class T>
int deserialize(Engine<Dim, T, CompressibleBrick>& engine, Stream& s){

  typedef Engine<Dim,T,CompressibleBrick> Engine_t;
  int nBytes=0;

  // Deserialize the domain interval.
  Interval<Dim> dom;
  nBytes+= deserialize(dom,s);

  // Deserialize the element type attribute.
  int totalSize;
  std::string elemType;
  nBytes+= deserialize(elemType,s);

  // Insist the element type match stored type.
  std::string thisElemType= PoomaCTTI(T());

  PInsist((elemType==thisElemType),
  "Error: source and target engine element types do not match.");

  // Initialize the engine with the restored interval domain
  engine= Engine_t(dom);

  // Read the compression flag and the data size.
  int compressed;
  int dataSize;
  nBytes+= deserialize(compressed,s);
  nBytes+= deserialize(dataSize,s);

  // Insist that the interval size accommodate the data size.
  int intervalSize= dom.size();
  PInsist((dataSize<=intervalSize),
  "Error: source data size too big for target interval.");

  // Get the data block. (This may or may not be compressed.)
  CompressibleBlock<T> cblock= engine.cblock();

  // Check the compression flag and prepare the data
  // target accordingly.
  T* pdata;
  if(compressed==1){
    // Get a pointer to the compressed value.
    pdata= &engine.compressedReadWrite();
  }
  else{
     // uncompress the block
    cblock.uncompress();
    // Get a pointer to the data block.
    pdata= cblock.data();
  }

  // Read the data.
  nBytes+= deserializeN(pdata,s,dataSize);

  return nBytes;
}

template <int Dim, class T>
int serialSizeof(const Engine<Dim, T, CompressibleBrick>& engine);
template <int Dim, class T>
int serialSizeof(const Engine<Dim, T, CompressibleBrick>& engine){

  int nBytes=0;

  // Obtain the attributes of the engine.

  // Get the Interval domain serial size.
  Interval<Dim> dom= engine.domain();
  nBytes+= serialSizeof(dom);

  // Get the total CompressibleBrick size.
  int totalSize= dom.size();

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Check the compression flag and set the data
  // parameters accordingly.
  int compressed;
  int dataSize;
  T* pdata;
  if(engine.compressed()){
    compressed= 1;
    dataSize= 1;
    // Get a pointer to the compressed value.
    pdata= &engine.compressedReadWrite();
  }
  else{
    compressed= 0;
    dataSize= totalSize;
    // Get the uncompressed data block.
    CompressibleBlock<T> cblock= engine.cblock();
    // Get a pointer to the data.
    pdata= cblock.data();
  }

  // Get the size of the descriptive attributes.
  nBytes+= 2*sizeof(int); // dataSize and compression
  nBytes+= serialSizeof(elemType);

  // Safest size computation obtains the size of
  // each element.
  for(int i=0;i<dataSize;i++){
    nBytes+= serialSizeof(pdata[i]);
  }
  return nBytes;
}

//-----------------------------------------------------------
// Serializers for
//   Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >
//-----------------------------------------------------------

template <class Stream, int Dim, class T, class LayoutTag, class PatchTag>
int serialize(Stream& s,
	      const Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
	      engine);
template <class Stream, int Dim, class T, class LayoutTag, class PatchTag>
int serialize(Stream& s,
	      const Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
	      engine){

  int nBytes=0;
  // Convenient typedefs.
  typedef Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> > Engine_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Layout_t::List_t List_t;
  typedef typename Layout_t::Value_t Node_t;

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Write the element type attribute.
  nBytes+= serialize(s,elemType);

  // Obtain the other attributes of the engine.
  // Get the layout object and serialize.
  Layout_t layout= engine.layout();
  nBytes+= serialize(s,layout);

  // Get the list of local nodes from the layout.
  List_t nodeList= layout.nodeListLocal();

  // Iterate through the local nodes and store the
  // patch engines.
  typename Engine_t::PatchID_t ID;
  typename Engine_t::PatchEngine_t patchEngine;
  typename List_t::iterator start= nodeList.begin(),
    end= nodeList.end();
  Node_t* node;
  for(;start!= end; ++start){
    node= *start;
    ID= node->localID();
    patchEngine= engine.localPatch(ID);
    nBytes+= serialize(s,patchEngine);
  }

  return nBytes;
}

template <class Stream, int Dim, class T, class LayoutTag, class PatchTag>
int deserialize(Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
		engine, Stream& s);
template <class Stream, int Dim, class T, class LayoutTag, class PatchTag>
int deserialize(Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
		engine, Stream& s){

  int nBytes=0;
  // Convenient typedefs.
  typedef Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> > Engine_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Layout_t::List_t List_t;
  typedef typename Layout_t::Value_t Node_t;

  // Read the element type attribute.
  std::string elemType;
  nBytes+= deserialize(elemType,s);

  // Insist the element type match stored type.
  std::string thisElemType= PoomaCTTI(T());

  PInsist((elemType==thisElemType),
  "Error: source and target engine element types do not match.");

  // Read the attributes of the stored engine.
  // Read the layout object.
  Layout_t layout;
  nBytes+= deserialize(layout,s);

  // Initialize the target multipatch engine using the layout.
  engine= Engine_t(layout);

  // Get the list of local nodes from the layout.
  List_t nodeList= layout.nodeListLocal();

  // Iterate through the local nodes and store the
  // patch engines.
  typename Engine_t::PatchID_t ID;
  typename Engine_t::PatchEngine_t patchEngine;
  typename List_t::iterator start= nodeList.begin(),
    end= nodeList.end();
  Node_t* node;
  for(;start!= end; ++start){
    node= *start;
    ID= node->localID();
    nBytes+= deserialize(patchEngine,s);
    engine.localPatch(ID)= patchEngine;
  }

  return nBytes;
}

template <int Dim, class T, class LayoutTag, class PatchTag>
int serialSizeof(const Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
engine);
template <int Dim, class T, class LayoutTag, class PatchTag>
int serialSizeof(const Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >&
engine){

  int nBytes=0;
  // Convenient typedefs.
  typedef Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> > Engine_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Layout_t::List_t List_t;
  typedef typename Layout_t::Value_t Node_t;

  // Construct an element type string.
  std::string elemType= PoomaCTTI(T());

  // Size the element type string.
  nBytes+= serialSizeof(elemType);

  // Obtain the other attributes of the engine.
  // Get the layout object and size.
  Layout_t layout= engine.layout();
  nBytes+= serialSizeof(layout);

  // Get the list of local nodes from the layout.
  List_t nodeList= layout.nodeListLocal();

  // Iterate through the local nodes and size the
  // patch engines.
  typename Engine_t::PatchID_t ID;
  typename Engine_t::PatchEngine_t patchEngine;
  typename List_t::iterator start= nodeList.begin(),
    end= nodeList.end();
  Node_t* node;
  for(;start!= end; ++start){
    node= *start;
    ID= node->localID();
    patchEngine= engine.localPatch(ID);
    nBytes+= serialSizeof(patchEngine);
  }

  return nBytes;
}

#endif // POOMA_IO_ENGINE_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: EngineIO.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/07/20 20:47:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo


