// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support Pooma Domains
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_DOMAIN_IO_H
#define POOMA_IO_DOMAIN_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "Domain/Loc.h"
#include "Domain/Grid.h"
#include "Domain/Domain.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include <string>
#if POOMA_NO_STRINGSTREAM
#include <strstream>
#else
#include <sstream>
#endif
//-----------------------------------------------------------
// I/O Functions for Loc<Dim>
//-----------------------------------------------------------
// CTTI function for Loc<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const Loc<Dim>&);
template <int Dim>
std::string PoomaCTTI(const Loc<Dim>&){
#if POOMA_NO_STRINGSTREAM
  std::strstream ss;
#else
  std::stringstream ss;
#endif

  int d=Dim;
  ss<<"Loc<"<<d<<">";
  return ss.str();
}

//-----------------------------------------------------------
// Serializers for Loc<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const Loc<Dim>& loc);
template <class Stream, int Dim>
int serialize(Stream& s, const Loc<Dim>& loc){
  int nBytes=0;
  // Write the dimension of the loc.
  int dim= Dim;
  nBytes+= serialize(s,dim);
  // Use the domain interface of the object to
  // get and serialize the values for each dimension.

  for(int i=0; i<dim; i++){
    nBytes+= serialize(s,loc[i].first());
  }
  return nBytes;
}

template <class Stream, int Dim>
int deserialize(Loc<Dim>& loc, Stream& s);
template <class Stream, int Dim>
int deserialize(Loc<Dim>& loc, Stream& s){

  int nBytes=0;
  // get the dimension
  int dim;
  int idim= Dim;
  nBytes+= deserialize(dim,s);

  // Insist the dimensions match.
  PInsist((dim==idim),
  "Error: source and target Loc dimensions do not match");

  // Initialize each dimension using the stored values.
  int first;
  for(int i=0;i<dim;i++){
    nBytes+= deserialize(first,s);
    loc[i]= Loc<1>(first);
  }

  return nBytes;
}

template <int Dim>
int serialSizeof(const Loc<Dim>& loc);
template <int Dim>
int serialSizeof(const Loc<Dim>& loc){
  return (1+Dim)*sizeof(int);
}


//-----------------------------------------------------------
// I/O Functions for Interval<Dim>
//-----------------------------------------------------------
// CTTI function for Interval<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const Interval<Dim>&);
template <int Dim>
std::string PoomaCTTI(const Interval<Dim>&){
#if POOMA_NO_STRINGSTREAM
  std::strstream ss;
#else
  std::stringstream ss;
#endif

  ss<<"Interval<"<<Dim<<">";
  return ss.str();
}

//-----------------------------------------------------------
// Serializers for Interval<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const Interval<Dim>& interval);
template <class Stream, int Dim>
int serialize(Stream& s, const Interval<Dim>& interval){
  int nBytes=0;
  // Write the dimension of the interval.
  int dim= Dim;
  nBytes+= serialize(s,dim);
  // Use the domain interface of the interval to
  // get and serialize first and last values for all dimensions.

  for(int i=0; i<dim; i++){
    nBytes+= serialize(s,interval[i].first());
    nBytes+= serialize(s,interval[i].last());
  }
  return nBytes;
}

template <class Stream, int Dim>
int deserialize(Interval<Dim>& interval, Stream& s);
template <class Stream, int Dim>
int deserialize(Interval<Dim>& interval, Stream& s){

  int nBytes=0;
  // get the dimension
  int dim;
  int idim= Dim;
  nBytes+= deserialize(dim,s);

  // Insist the dimensions match.
  PInsist((dim==idim),
  "Error: source and target interval dimensions do not match");

  // Initialize each dimension using the stored bounds.
  int first,last;
  for(int i=0;i<dim;i++){
    nBytes+= deserialize(first,s);
    nBytes+= deserialize(last,s);
    interval[i]= Interval<1>(first,last);
  }

  return nBytes;
}

template <int Dim>
int serialSizeof(const Interval<Dim>& interval);
template <int Dim>
int serialSizeof(const Interval<Dim>& interval){
  return (1+2*Dim)*sizeof(int);
}

#endif // POOMA_IO_DOMAIN_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DomainIO.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/07/20 20:47:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo





