// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   DiscreteGeometry
//   NewGeometry + specializations
//   NewGeometryDomain + specializations
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_DISCRETEGEOMETRY_H
#define POOMA_GEOMETRY_DISCRETEGEOMETRY_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Layout/INode.h"

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

template<int Dim> class Range;
template<int Dim, int SliceDim> class SliceInterval;
template<int Dim, int SliceDim> class SliceRange;

template<class Mesh> class MeshView;

template<int Dim> class NoGeometry;

//-----------------------------------------------------------------------------
// Overview: 
// 
// DiscreteGeometry:  Base template for classes that tie together a centering
//            and a mesh to provide geometry services for a field.
//
// To model the DiscreteGeometry concept, classes must provide the following:
//
// Exported typedefs:
//
//   Centering_t - the centering tag.
//   CoordinateSystem_t - the coordinate system (obtained from the mesh).
//   Domain_t - the types of the physical/total domains (obtained from the 
//     mesh).
//   GuardLayers_t - the type of GuardLayers object for this geometry.
//   PointType_t - the type representing points in the geometry.
//   PositionArray_t - the type of constArray returned from x().
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the mesh.
//   componentCentered - true if this field has different centerings
//     for each component.
//   coordinateDimensions - the dimension of the coordinate system.
//
// Accessors:
//
//   boundingBox() - returns a Region giving the bounding box of this
//     geometry.
//   centering() - returns the centering object for this geometry.
//   guardLayers() - returns the GuardLayers object for this geometry.
//   initialized() - returns whether or not a mesh has been initialized.
//   mesh() - returns the mesh.
//   physicalDomain() - returns this geometry's physical domain; that is, the
//     domain that is used to define a field (excludes global guard layers).
//   totalDomain() - returns this geometry's total domain (includes global 
//     guard layers).
//   x() - returns an array of centering positions corresponding to the
//     total domain.
//-----------------------------------------------------------------------------

template<class Centering, class Mesh>
class DiscreteGeometry;


//-----------------------------------------------------------------------------
// Full Description:
// NewGeometry<Geometry, SubDomain>
//
// Traits class for determining the type obtained by subsetting a 
// particular Geometry with a particular SubDomain.
//-----------------------------------------------------------------------------

template <class Geometry, class SubDomain>
struct NewGeometry;

// Specializations that cover all the cases for POOMA 2.1:

template<class Centering, class Mesh, int Dim>
struct NewGeometry<DiscreteGeometry<Centering, Mesh>, Interval<Dim> >
{
  typedef DiscreteGeometry<Centering, MeshView<Mesh> > Type_t;
};

template<class Centering, class Mesh, int Dim>
struct NewGeometry<DiscreteGeometry<Centering, Mesh>, INode<Dim> >
{
  typedef DiscreteGeometry<Centering, MeshView<Mesh> > Type_t;
};

template<class Centering, class Mesh, int Dim>
struct NewGeometry<DiscreteGeometry<Centering, Mesh>, Range<Dim> >
{
  typedef NoGeometry<Dim> Type_t;
};

template<class Centering, class Mesh, int Dim, int SliceDim>
struct NewGeometry<
  DiscreteGeometry<Centering, Mesh>, SliceInterval<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

template<class Centering, class Mesh, int Dim, int SliceDim>
struct NewGeometry<
  DiscreteGeometry<Centering, Mesh>, SliceRange<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

template<class Centering, class Mesh, int Dim>
struct NewGeometry<
  DiscreteGeometry<Centering, MeshView<Mesh> >, Interval<Dim> >
{
  typedef DiscreteGeometry<Centering, MeshView<Mesh> > Type_t;
};

template<class Centering, class Mesh, int Dim>
struct NewGeometry<
  DiscreteGeometry<Centering, MeshView<Mesh> >, INode<Dim> >
{
  typedef DiscreteGeometry<Centering, MeshView<Mesh> > Type_t;
};

template<class Centering, class Mesh, int Dim>
struct NewGeometry<
  DiscreteGeometry<Centering, MeshView<Mesh> >, Range<Dim> >
{
  typedef NoGeometry<Dim> Type_t;
};

template<class Centering, class Mesh, int Dim, int SliceDim>
struct NewGeometry<
  DiscreteGeometry<Centering, MeshView<Mesh> >, SliceInterval<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

template<class Centering, class Mesh, int Dim, int SliceDim>
struct NewGeometry<
  DiscreteGeometry<Centering, MeshView<Mesh> >, SliceRange<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

template<int Dim>
struct NewGeometry<NoGeometry<Dim>, INode<Dim> >
{
  typedef NoGeometry<Dim> Type_t;
};

template<int Dim>
struct NewGeometry<NoGeometry<Dim>, Interval<Dim> >
{
  typedef NoGeometry<Dim> Type_t;
};

template<int Dim>
struct NewGeometry<NoGeometry<Dim>, Range<Dim> >
{
  typedef NoGeometry<Dim> Type_t;
};

template<int Dim, int SliceDim>
struct NewGeometry<NoGeometry<Dim>, SliceInterval<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

template<int Dim, int SliceDim>
struct NewGeometry<NoGeometry<Dim>, SliceRange<Dim, SliceDim> >
{
  typedef NoGeometry<SliceDim> Type_t;
};

//-----------------------------------------------------------------------------
// Full Description:
// NewGeometryGeometry<Geometry, SubDomain>
//
// This traits class allows you to modify the geometry
// that gets passed to the view geometry that results from a subset
// operation.
//-----------------------------------------------------------------------------

template <class Geometry, class SubDomain>
struct NewGeometryGeometry
{
  typedef Geometry Type_t;
  typedef const Geometry &Return_t;
  static inline
  Return_t apply(const Geometry &g, const SubDomain &)
  {
    return g;
  }
};

//-----------------------------------------------------------------------------
// Full Description:
// NewGeometryDomain<Geometry, SubDomain>
//
// This traits class allows you to modify the domain
// that gets passed to the view geometry that results from a subset
// operation.
//-----------------------------------------------------------------------------

template <class Geometry, class SubDomain>
struct NewGeometryDomain
{
  typedef SubDomain Type_t;
  typedef const SubDomain &Return_t;
  static inline
  Return_t apply(const Geometry &, const SubDomain &i)
  {
    return i;
  }
};

// Specializations for INodes:

template<class Centering, class Mesh, int Dim>
struct NewGeometryDomain<DiscreteGeometry<Centering, Mesh>, INode<Dim> >
{
  typedef Interval<Dim> Type_t;
  typedef const Interval<Dim> &Return_t;
  static inline
  Return_t apply(const DiscreteGeometry<Centering, Mesh> &, 
    const INode<Dim> &i)
  {
    return i.domain();
  }
};

template<int Dim>
struct NewGeometryDomain<NoGeometry<Dim>, INode<Dim> >
{
  typedef Interval<Dim> Type_t;
  typedef const Interval<Dim> &Return_t;
  static inline
  Return_t apply(const NoGeometry<Dim> &, const INode<Dim> &i)
  {
    return i.domain();
  }
};

#endif // POOMA_GEOMETRY_DISCRETEGEOMETRY_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DiscreteGeometry.h,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/03/07 13:17:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
