// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Field test 1: Basic stuff: declaring, viewing, and indexing fields.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

//-----------------------------------------------------------------------------
// Globals
//-----------------------------------------------------------------------------

const int NX = 5, NY = 5;
Interval<1> I(NX), J(NY);
Interval<2> td(Interval<1>(-1,NX-1), Interval<1>(-1,NY-1)), pd(NX-1, NY-1);
  

//-----------------------------------------------------------------------------
// Test function
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void doTest(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f)
{
  tester.check("PD", f.physicalDomain(), pd);
  tester.check("TD", f.totalDomain(), td);
    
  f.addBoundaryConditions(AllZeroFaceBC());
    
  for (int i = 0; i <= f.physicalDomain()[0].last(); i++)
    for (int j = 0; j <= f.physicalDomain()[1].last(); j++)
      f(i, j) = i + j;
      
  f.applyBoundaryConditions();
  
  Pooma::blockAndEvaluate();
       
  tester.out() << f << std::endl;
  tester.out() << f.read() << std::endl;
  tester.out() << f() << std::endl;
  
  tester.check("f(4,4)", f(4,4), 0.0);
  tester.check("f.read(4,4)", f.read(4,4), 0.0);
  tester.check("f(0,3)", f(0,3), 3.0);
  
  Range<1> R(0,2,2);
  Range<2> RR(R, R);
  tester.out() << f.x() << std::endl;
  tester.out() << f.x(RR) << std::endl;
  
  Loc<2> loc(2, 3);
  f(loc) = 1.0;
  tester.out() << f << std::endl;
  
  tester.out() << f(R, R) << std::endl;
  tester.check("sum(f(R,R))", sum(f(R,R)), 8.0);
}


//-----------------------------------------------------------------------------
// Main program
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh.
  
  Vector<2,double> origin;
  Vector<2,double> spacings;
  for (int d = 0; d < 2; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  UniformRectilinearMesh<2> mesh(I, J, origin, spacings);
  
  // Create the geometry.
  
  typedef DiscreteGeometry<Cell, UniformRectilinearMesh<2> > Geometry_t;
  
  Geometry_t geom(mesh, GuardLayers<2>(1));
  
  // Make a Brick-Engine-based field.
  
  Field<Geometry_t> f(geom);

  doTest(tester, f);
  
  // Make a CompressibleBrick-Engine-based field.
  
  Field<Geometry_t, double, CompressibleBrick> fc(geom);

  doTest(tester, fc);
  
  // Make a Multipatch-Engine-based field.

  Loc<2> blocks(2,2);
  UniformGridLayout<2> layout(pd, blocks, GuardLayers<2>(0), 
                              GuardLayers<2>(1), ReplicatedTag());
  typedef MultiPatch<UniformTag, Brick> MP_t;
  Field<Geometry_t, double, MP_t> fm(geom, layout);

  doTest(tester, fm);
  
  // Make a Nonuniform Multipatch-Engine-based field.

  GridLayout<2> layout2(pd, blocks, GuardLayers<2>(0), GuardLayers<2>(1),
                        ReplicatedTag());
  typedef MultiPatch<GridTag, Brick> MP2_t;
  Field<Geometry_t, double, MP2_t> fg(geom, layout2);

  doTest(tester, fg);
    
  int ret = tester.results("field_test1");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test1.cpp,v $   $Author: julianc $
// $Revision: 1.11 $   $Date: 2000/06/07 16:25:48 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
