// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// DynamicLayout Dynamic Ops test code
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Domain/Loc.h"
#include "Domain/Interval.h"
#include "Layout/DynamicLayout.h"
#include "Partition/GridPartition.h"
#include "Partition/UniformMapper.h"
#include "Engine/DynamicEngine.h"
#include "Engine/RemoteDynamicEngine.h"
#include "Engine/MultiPatchEngine.h"
#include "Array/Array.h"

#include <iostream>
#include <vector>
using namespace std;

#ifdef POOMA_CHEETAH
typedef MultiPatch<DynamicTag,Remote<Dynamic> > DynamicMultiPatch_t;
#else
typedef MultiPatch<DynamicTag,Dynamic> DynamicMultiPatch_t;
#endif

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);

  // Create the total domain.

  int numContexts = Pooma::contexts();
  Interval<1> x(12*numContexts);
  Interval<1> domain(x);
  
  // Create the block sizes.
  
  Loc<1> blocks(3*numContexts), blocks2(4*numContexts);

  // Create the partitioners.
  
  GridPartition<1> partition(blocks), partition2(blocks2);

  // Create the layouts.
  
  DynamicLayout layout(domain, partition, UniformMapper(partition));
  DynamicLayout layout2(domain, partition2, UniformMapper(partition2));

  tester.out() << "layout1 = " << layout << endl;
  tester.out() << "layout2 = " << layout2 << endl;

  Array<1, double, DynamicMultiPatch_t> a(layout);
  Array<1, double, DynamicMultiPatch_t> aa(layout);
  Array<1, double, DynamicMultiPatch_t> a2(layout2);
  
  Array<1, double, DynamicMultiPatch_t> g(layout);
  Array<1, double, DynamicMultiPatch_t> gg(layout);
  Array<1, double, DynamicMultiPatch_t> g2(layout2);  

  typedef DynamicEvents::PatchID_t             PatchID_t;
  typedef DynamicEvents::CreateSize_t          CreateSize_t;



  // Store some stuff.
  
  typedef Interval<1>::iterator Iterator;
  Iterator b0 = domain[0].begin();
  Iterator e0 = domain[0].end();
  for (Iterator i0=b0; i0!=e0; ++i0)
  {
    int j=i0->first();
     a(j) = aa(j)  = g(j) = gg(j)  = double(j);
     a2(j)=g2(j)=double(j);
  }

  tester.out() << " On layout1, Array a = " << a << endl;

  tester.out() << " destroy element 1 on patch 0 " << endl;

  PatchID_t zero = 0;
  PatchID_t one = 1;
  layout.destroy(Interval<1>(1,1),zero,BackFill());
  layout.sync();

  tester.out() << " after destroy, this is array a " <<endl;

  tester.out()  << a << endl;
  
  tester.out() << " this is array aa, should be the same as a " <<endl;

  tester.out() << aa << endl;
  
  tester.out() << " copy elements 0 and 1 from patch 1 to patch 0. " << endl;

  layout.copy(Interval<1>(0,1),one,zero);

  tester.out() << " after copy, before sync"<<endl;
  
  layout.sync();

  tester.out() << " after sync, this is array a " <<endl;

  tester.out()  << a << endl;
  
  tester.out() << " this is array aa, should be the same as a " <<endl;

  tester.out() << aa << endl;
  
  tester.check(sum((aa - a)*(aa - a)) == 0);

  tester.out() << " create 5 in patch 2 " <<endl;

  layout.create(5,2);
  layout.sync();
    
  for (int i=9;i<=layout.domain().last();++i)
  {
    a(i) = double(i);
    aa(i) = 100 * a(i);
  }

  tester.out() << " this is array a " <<endl;

  tester.out()  << a << endl;

  
  tester.out() << " this is array aa, should be the same as a, "
               << "but with values in patch 2 multiplied by 100." <<endl;


  tester.out() << aa << endl;

#if 0
  // The following are dynamic operations with global domains. They do
  // not work cross context, and it is not clear that we even want
  // this capability, particularly the copy, which mixes global
  // domains and local PatchIDs in a really bad manner.

  tester.out() << " do cross-patch destroy, deleting from 3 to last-2 " <<endl;

  Interval<1> crosspatch(3,layout.domain().last()-2);
  layout.destroy(crosspatch,-1,BackFill());
  layout.sync();
  
  tester.out() << " this is array a " <<endl;

  tester.out()  << a << endl;

  tester.out() << " Array a2, using layout2 " <<a2<<endl;

  tester.out() << "Copy even elements to end " <<endl;

  Range<1> cpcopy(0,11,2);

  layout2.copy(cpcopy,-1);
  
  layout2.sync();

  tester.out() << " Array a2 " <<a2<<endl;

  Array<1, double, DynamicMultiPatch_t> copyg2(g2);
  
  tester.out() << " this is a copy of g2, called copyg2"<<endl;
  
  tester.out() << copyg2<<endl;
  
  layout2.destroy(crosspatch,-1,ShiftUp());
  
  layout2.sync();
  
  tester.out() << " after kill, shiftup on "<<crosspatch<<endl;
  
  tester.out() << " copyg2 "<<copyg2<<endl;
#endif

  layout2.create(6);
  layout2.sync();

  tester.out() << " a.domain() = " <<  a.domain() << std::endl;
  tester.out() << "aa.domain() = " << aa.domain() << std::endl;
  tester.out() << "a2.domain() = " << a2.domain() << std::endl;

  a2 = aa + a * 0.1;

  int ret = tester.results("dynamiclayout_test1");
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dynamiclayout_test1.cpp,v $   $Author: jac $
// $Revision: 1.5 $   $Date: 2000/06/06 20:46:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
