// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// EnginePatch          - functor for getting the nth patch from an engine.
// EngineNumPatches     - gives the number of patches in an engine.
// EnginePatchEmpty     - tells you if the nth patch is empty.
// PatchView<Container> - traits class for the type of object returned
//                        by the patch() member function.
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_ENGINEPATCH_H
#define POOMA_ENGINE_ENGINEPATCH_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"
#include "Pooma/PETE/AssertEquals.h"
#include "Pooma/View.h"
#include "Engine/EngineFunctor.h"

//-----------------------------------------------------------------------------
//
// EnginePatch
//
// EnginePatch is a tag used with engineFunctor to return the ith patch
// from a multi-patch engine in a generic way.  Non-multipatch engines
// are defined to have 1 patch, so you can use EnginePatch on them as well.
// The syntax looks like:
//
// patchEngine = engineFunctor(engine, EnginePatch(i));
//
//-----------------------------------------------------------------------------

struct EnginePatch
{
  // Expression engines need to be combine the patch engines into
  // a new tree:
  typedef TreeCombine Combine_t;
  typedef int PatchID_t;

  explicit EnginePatch(PatchID_t patch) : patch_m(patch) { }
  PatchID_t patch_m;
};


template<class Eng>
struct EngineFunctorDefault<Eng, EnginePatch>
{
  typedef Eng Type_t;

  inline static
  const Type_t &apply(const Eng &e, const EnginePatch &)
  {
    // Engines that are multipatch must specialize this functor
    // to access the patch
    CTAssert(!(Eng::multiPatch));
    return e;
  }
};

template<class T>
struct LeafFunctor<Scalar<T>, EnginePatch>
{
  typedef Scalar<T> Type_t;

  inline static
  Type_t apply(const Scalar<T> &scalar, const EnginePatch &)
  {
    return scalar;
  }
};

//-----------------------------------------------------------------------------
//
// PatchView<Container>
//
// PatchView is the traits class used to describe the result of
// .patch(i) for a container class.  PatchView is specialized for
// Arrays etc. to give the Array with the engine given by
// EngineFunctor<Engine,EnginePatch>::Type_t;
//
//-----------------------------------------------------------------------------

template<class Container>
struct Patch
{
};


template<class Container>
struct PatchView
{
  typedef typename Patch<Container>::Type_t Patch_t;
  typedef typename Patch_t::Domain_t Dom_t;
  typedef typename View1<Patch_t, Dom_t>::Type_t Type_t;

  inline static
  Type_t make(const Container &subject, int i)
  {
    return subject.patchLocal(i)();
  }
};

template<class Node>
struct LeafFunctor<Node, EnginePatch>
{
  typedef typename PatchView<Node>::Type_t Type_t;

  inline static
  Type_t apply(const Node &node, const EnginePatch &tag)
  {
    return node.patchLocal(tag.patch_m)();
  }
};

//-----------------------------------------------------------------------------
// 
// EngineNumPatches
//
// EngineNumPatches is used to find out how many patches an engine has.
// (Or throw an assertion if the answer is ambiguous.)  Typical use would
// look like:
//
// int n = engineFunctor(a.engine(), EngineNumPatches());
// for (i = 0; i < n; ++i)
//     calculate(a.patch(i));
//
//-----------------------------------------------------------------------------

struct EngineNumPatches
{
  // Throw an assertion if not all the engines have the same number of
  // patches:
  typedef AssertEquals Combine_t;
};

// Generic engines have 1 patch.

template<class Eng>
struct EngineFunctorDefault<Eng, EngineNumPatches>
{
  typedef int Type_t;

  inline static
  Type_t apply(const Eng &, const EngineNumPatches &)
  {
    // Engines that are multipatch must specialize this functor
    // to access the patch
    CTAssert(!(Eng::multiPatch));
    return 1;
  }
};

// Scalars have 0 patches.

template<class T>
struct EngineFunctorScalar<T, EngineNumPatches>
{
  typedef int Type_t;

  inline static
  Type_t apply(const T &, const EngineNumPatches &)
  {
    return 0;
  }
};

/////////////////////////////////////////////////////////////////////

#endif     // POOMA_ENGINE_ENGINEPATCH_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: EnginePatch.h,v $   $Author: swhaney $
// $Revision: 1.15 $   $Date: 2000/07/20 15:41:36 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
