// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Classes: 
//   ConstantFunction      - Tag class for defining a constant-function-engine.
//   Engine                - Specialization for ConstantFunction
//   NewEngine             - Specializations for ConstantFunction
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_CONSTANTFUNCTIONENGINE_H
#define POOMA_ENGINE_CONSTANTFUNCTIONENGINE_H

//-----------------------------------------------------------------------------
// Overview:
//
// Constant-function-engine objects provide a way to make a scalar behave like
// an array.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/SliceDomain.h"
#include "Engine/Engine.h"
#include "Layout/INode.h"
#include "PETE/ErrorType.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// ConstantFunction is just a tag class for the constant-function engine,
// which makes a scalar look like an Array.
//-----------------------------------------------------------------------------

struct ConstantFunction 
{ };

//-----------------------------------------------------------------------------
// Full Description:
// 
// Engine<Dim, T, ConstantFunction> is a specialization of Engine for 
// ConstantFunction.
//
// This does all of the usual Engine things:
// 
// Typedefs for the tag, element types, domain and dimensions.
// Operator() with integers to evaluate elements quickly.
// Operator() with a domain to subset.
// Accessor for the domain.
//-----------------------------------------------------------------------------

template<int Dim, class T>
class Engine<Dim, T, ConstantFunction>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef ConstantFunction                   Tag_t;
  typedef Engine<Dim, T, ConstantFunction>   This_t;
  typedef This_t                             Engine_t;
  typedef Interval<Dim>                      Domain_t;
  typedef T                                  Element_t;
  typedef ErrorType                          ElementRef_t;

  enum { dimensions = Dim };
  enum { hasDataObject = false };
  enum { dynamic = false };
  enum { zeroBased = false };
  enum { multiPatch = false };

  //---------------------------------------------------------------------------
  // Construct from a domain object.

  Engine(const Domain_t &domain, T val = T())
  : domain_m(domain), val_m(val) 
  { 
    for (int d = 0; d < Dim; ++d)
      firsts_m[d] = domain[d].first();
  }

  //---------------------------------------------------------------------------
  // Copy constructor.

  Engine(const Engine<Dim, T, ConstantFunction> &model)
  : val_m(model.constant()), domain_m(model.domain())
  {
    for (int d = 0; d < Dim; ++d)
      {
        firsts_m[d] = model.firsts_m[d];
      }
  }

  //---------------------------------------------------------------------------
  // Construct from various sorts of domains (e.g., take a view).

  template<class DT>
  Engine(const Engine<Dim, T, ConstantFunction> &e, const Domain<Dim, DT> &dom)
  : val_m(e.constant()), domain_m(Pooma::NoInit())
  {
    const typename DT::Domain_t &domain = dom.unwrap();
    for (int d = 0; d < Dim; ++d)
      {
        domain_m[d]  = Interval<1>(domain[d].length());
        firsts_m[d] = 0;
      }
  }

  template<int Dim2, class DT>
  Engine(const Engine<Dim2, T, ConstantFunction> &e, 
    const SliceDomain<DT> &dom)
  : val_m(e.constant()), domain_m(Pooma::NoInit())
  {
    // The domain's dimension should match ours.
    
    CTAssert(DT::sliceDimensions == Dim);
    CTAssert(DT::dimensions == Dim2);

    const typename DT::SliceDomain_t &domain = dom.sliceDomain();
    for (int d = 0; d < Dim; ++d)
      {
        domain_m[d]  = Interval<1>(domain[d].length());
        firsts_m[d] = 0;
      }
  }

  template<class Domain>
  Engine(const Engine<Dim, T, ConstantFunction> &e, const Node<Domain> &node)
  : val_m(e.constant()), domain_m(Pooma::NoInit())
  {
    // The nodes's dimension should match ours.
    
    CTAssert(Domain::dimensions == Dim);

    const Domain &domain = node.domain();
    for (int d = 0; d < Dim; ++d)
      {
        domain_m[d]  = Interval<1>(domain[d].length());
        firsts_m[d] = 0;
      }
  }

  Engine(const Engine<Dim, T, ConstantFunction> &e, const INode<Dim> &inode)
  : val_m(e.constant()), domain_m(Pooma::NoInit())
  {
    const typename INode<Dim>::Domain_t &domain = inode.domain();
    for (int d = 0; d < Dim; ++d)
      {
        domain_m[d]  = Interval<1>(domain[d].length());
        firsts_m[d] = 0;
      }
  }
    
  //---------------------------------------------------------------------------
  // Element access via ints for speed.
  // We only need read() functions since this engine should only be used in
  // a read-only array.

  inline Element_t read(int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int, int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int, int, int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int, int, int, int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int, int, int, int, int) const 
    {
      return val_m;
    }
  inline Element_t read(int, int, int, int, int, int, int) const 
    {
      return val_m;
    }
  inline Element_t read(const Loc<Dim> &) const
    {
      return val_m;
    }

  //---------------------------------------------------------------------------
  // Return the domain.

  const Domain_t &domain() const { return domain_m; }

  //---------------------------------------------------------------------------
  // Return the first value for the specified direction.
  
  inline int first(int i) const
  {
    PAssert(i >= 0 && i < Dim);
    return firsts_m[i];
  }

  //---------------------------------------------------------------------------
  // Accessors/modifiers.

  T constant() const { return val_m; }
  void setConstant(T val) { val_m = val; }

private:

  T val_m;
  Domain_t domain_m;
  int firsts_m[Dim];
};

//-----------------------------------------------------------------------------
// Full Description:
// NewEngine<Engine,SubDomain>
//
// Specializations of NewEngine for subsetting a constant-function-engine with
// an arbitrary domain. 
//-----------------------------------------------------------------------------

template <int Dim, class T>
struct NewEngine<Engine<Dim, T, ConstantFunction>, Interval<Dim> >
{
  typedef Engine<Dim, T, ConstantFunction> Type_t;
};

template <int Dim, class T>
struct NewEngine<Engine<Dim, T, ConstantFunction>, Range<Dim> >
{
  typedef Engine<Dim, T, ConstantFunction> Type_t;
};

template <int Dim, class T, int SliceDim>
struct NewEngine<Engine<Dim,T,ConstantFunction>, SliceInterval<Dim,SliceDim> >
{
  typedef Engine<SliceDim,T,ConstantFunction> Type_t;
};

template <int Dim, class T, int SliceDim>
struct NewEngine<Engine<Dim,T,ConstantFunction>, SliceRange<Dim,SliceDim> >
{
  typedef Engine<SliceDim,T,ConstantFunction> Type_t;
};

template <int Dim, class T, class Domain>
struct NewEngine<Engine<Dim, T, ConstantFunction>, Node<Domain> >
{
  typedef Engine<Dim, T, ConstantFunction> Type_t;
};

template <int Dim, class T>
struct NewEngine<Engine<Dim, T, ConstantFunction>, INode<Dim> >
{
  typedef Engine<Dim, T, ConstantFunction> Type_t;
};

#endif // POOMA_ENGINE_CONSTANTFUNCTIONENGINE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ConstantFunctionEngine.h,v $   $Author: swhaney $
// $Revision: 1.16 $   $Date: 2000/07/20 15:39:26 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
