// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// DynamicArray bad create: Try create on an DynamicArray with existing view
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Pooma/DynamicArrays.h"

// a test function that takes as arguments:
//   1. A DynamicArray
//   2. An array which is a view on the DynamicArray

template<class T, class CA>
bool
testview(Pooma::Tester &tester, DynamicArray<T,Dynamic> &da,
	 const CA &daview)
{
  tester.out() << "In testview:" << std::endl;
  tester.out() << "    da = " << da << std::endl;
  tester.out() << "daview = " << daview << std::endl;

  // the following should crash
  tester.out() << "Trying to create values within da ..." << std::endl;
  bool result = false;
#if POOMA_EXCEPTIONS
  try {
    da.create(3);
    tester.out() << "Ack! create call didn't throw!!!" << std::endl;
    result = false;
  }
  catch(const Pooma::Assertion &a)
  {
    tester.out() << "Caught assertion - it worked!" << std::endl;
    result = true;
  }
#else
  da.create(3);
  tester.out() << "Ack! Program should have aborted and never gotten here!"
               << std::endl;
#endif

  return result;
}


int main(int argc, char *argv[])
{
  // Initialize POOMA and output stream, using Tester class

  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);
  tester.out() << argv[0];
  tester.out() << ": DynamicArray dynamic ops w/views." << std::endl;
  tester.out() << "-------------------------------------------" << std::endl;

  // Create an Interval object to create and index into an Array with

  tester.out() << "Creating an Interval<1> object ..." << std::endl;
  Interval<1> D1(3);
  tester.out() << "D1 = " << D1 << std::endl;

  // Create simple dynamic array.

  tester.out() << "Creating DynamicArray using domain ..." << std::endl;
  DynamicArray<int,Dynamic> a(D1);
  tester.check(a.domain().size() == D1.size());

  // Initialize dynamic array with scalar.

  a = 3;
  tester.out() << "Initialized DynamicArray to 3:" << std::endl;
  tester.out() << "a = " << a << std::endl;
  tester.check(sum(a) == (a.domain().size() * 3));

  // Create elements in the array.

  tester.out() << "Creating 2 elements at end of a ..." << std::endl;
  a.create(2);
  a.sync();
  tester.out() << "a = " << a << std::endl;
  tester.check(a.domain().size() == (D1.size() + 2));

  // Call a function which takes a view and the original DynamicArray

  std::cout << "The program should abort in the next operation when it\n";
  std::cout << "tries to create elements in an array with an existing view.";
  std::cout << std::endl;
  tester.out() << "Calling testview with a and a(1,3) ..." << std::endl;
  tester.check(testview(tester, a, a(Interval<1>(1,3))));

  // Return resulting error code and exit; Tester will shut down POOMA.

  tester.out() << "-------------------------------------------" << std::endl;
  int retval = tester.results("DynamicArray dynamic ops w/views");
  Pooma::finalize();
  return retval;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dynamic_array_badcreate.cpp,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/07/20 15:39:26 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
