// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_INDIRECTIONLIST_ITERATOR_H
#define POOMA_DOMAIN_INDIRECTIONLIST_ITERATOR_H

//-----------------------------------------------------------------------------
// Classes: 
//   IndirectionListIterator
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
//   IndirectionListIterator - Iterates through IndirectionList<T> elements.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/IndirectionList.h"
#include "Utilities/PAssert.h"

#include <iterator> // std::random_access_iterator_tag
#include <stddef.h> // ptrdiff_t

//-----------------------------------------------------------------------------
// Open POOMA namespace:
//-----------------------------------------------------------------------------

// namespace POOMA {

//-----------------------------------------------------------------------------
//
// Full Description:
// IndirectionListIterator
//
// A random access iterator class that iterates through all of the elements
// of an IndirectionList<T>, returning T's when dereferenced.
//
//-----------------------------------------------------------------------------

template <class T>
class IndirectionListIterator
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================

  typedef IndirectionListIterator<T>          This_t;
  typedef IndirectionList<T>                  Domain_t;
  typedef T                                   Value_t;

  typedef std::random_access_iterator_tag     iterator_category;
  typedef T                                   value_type;
  typedef ptrdiff_t                           difference_type;
  typedef const T*                            pointer;
  typedef const T&                            reference;

  //============================================================
  // Constructors
  //============================================================

  // The main IndirectionListIterator constructor stores the given domain
  // and initializes the iterator to the beginning.

  IndirectionListIterator(const Domain_t &d, int initial_pos = 0)
  : domain_m(d), pos_m(initial_pos)
  {
    if (!done())
      val_m = domain_m(pos_m);
  }
    
  // Copy constructor.

  IndirectionListIterator(const This_t &it)
  : domain_m(it.domain_m), pos_m(it.pos_m), val_m(it.val_m)
  { }

  // The default constructor constructs an end iterator for an empty domain.

  IndirectionListIterator() { }

  //============================================================
  // Accessors
  //============================================================

  // Dereference operator. Returns const ref to T.

  inline const Value_t & operator*() const { PAssert(!done()); return val_m; }

  // Member selection operator. 

  inline const Value_t * operator->() const { PAssert(!done()); return &val_m; }

  // Equality tests.
  // This only tests that the iterators have the same position.
  // It does not test whether the underlying domains are the same.
  
  inline bool operator==(const This_t &i) const { return pos_m == i.pos_m; }
  inline bool operator!=(const This_t &i) const { return pos_m != i.pos_m; }
  inline bool operator< (const This_t &i) const { return pos_m <  i.pos_m; }
  inline bool operator<=(const This_t &i) const { return pos_m <= i.pos_m; }
  inline bool operator> (const This_t &i) const { return pos_m >  i.pos_m; }
  inline bool operator>=(const This_t &i) const { return pos_m >= i.pos_m; }
  
  inline Value_t operator[](int n) const
  {
    return domain_m(pos_m + n);
  }
  
  This_t operator+(int n) const
  {
    This_t ret(*this);
    ret += n;
    return ret;
  }

  This_t operator-(int n) const
  {
    This_t ret(*this);
    ret -= n;
    return ret;
  }

  ptrdiff_t
  operator-(const This_t &it) const
  {
    //    PAssert(domain_m == it.domain_m);
    //    Cannot check this currently for IndirectionLists!
    return pos_m - it.pos_m;
  }
    
  // At-end (false) test.
  // Returns true if this iterator is at-end.

  bool done() const { return (pos_m > domain_m.size()-1); }

  //============================================================
  // Mutators
  //============================================================

  // Assignment operator.

  This_t &operator=(const This_t &it)
  {
    if (&it != this)
      {
        domain_m = it.domain_m;
        pos_m    = it.pos_m;
        val_m    = it.val_m;
      }
    return *this;
  }

  // Pre-increment operator. 

  This_t &operator++() { increment();   return *this; }
  This_t &operator--() { increment(-1); return *this; }

  // Post-increment operator.
  // This has to make a copy, so prefer the above if possible.

  This_t operator++(int)
  {
    This_t save(*this);
    increment();
    return save;
  }

  This_t operator--(int)
  {
    This_t save(*this);
    increment(-1);
    return save;
  }

  inline This_t operator+=(int n) { increment(n);  return *this; }
  inline This_t operator-=(int n) { increment(-n); return *this; }

private:

  //============================================================
  // Data members.
  //============================================================

  // The domain we're iterating over.

  Domain_t domain_m;

  // Our current position in the domain.
  
  int pos_m;
    
  // Our current value.
  
  Value_t val_m;
    
  //============================================================
  // Implementation functions
  //============================================================

  // Increment iterator.

  inline void increment()
  {
    PAssert(!done());
    ++pos_m;
    if (!done())
      val_m = domain_m(pos_m);
    else
      val_m = Value_t();
  }
  
  inline void increment(int n)
  {
    //    PAssert(!done()); 
    //    fails if we try to decrement from end, which is legal
    pos_m += n;
    if (!done())
      val_m = domain_m(pos_m);
    else
      val_m = Value_t();
  }
};

template <class T>
inline
IndirectionListIterator<T> operator+(int n,
  const IndirectionListIterator<T> &it)
{
  IndirectionListIterator<T> ret(it);
  ret += n;
  return ret;
}

// } // namespace POOMA

#endif // POOMA_DOMAIN_INDIRECTIONLIST_ITERATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: IndirectionListIterator.h,v $   $Author: jac $
// $Revision: 1.4 $   $Date: 2000/06/14 22:28:47 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
