// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_CONTAINS_H
#define POOMA_DOMAIN_CONTAINS_H

//-----------------------------------------------------------------------------
// Function:
//   bool contains(domain, domain);
// Class:
//   ContainsDomain<T1,T2,Dim>
//   ContainsDomainSingle<T1,T2,bool>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// bool contains(domain,domain) is a global function which determines if
// the points in the second domain are all points which are in the first
// domain.  If there is even just one point in the second not in the first,
// then this returns false.  Note that the order is important: if
// contains(a,b) is true, then the only way that contains(b,a) can be true is
// if a == b.  The order for the query is: is b contained within a?
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description of ContainsDomainSingle:
//
// ContainsDomainSingle<T1,T2,bool strided>::contains(a,b) compares two
// domains a and b of type T1 and T2, and returns true if a contains b.
// a and b are assumed to be 1D domains, and this struct is used by
// ContainsDomain for each dimension in a multidimensional contains operation.
// The final boolean template parameter is used to specialize the calculation
// to the following two cases:
//   strided == false: one or both of the domains has unit stride.  In
//                     this case, the computation is quite simple: check if
//                     the endpoints of b lie within the endpoints of a.
//   strided == true: neither domain has unit stride.  This is more complicated
//                    since it is possible that even if the endpoints of b
//                    are contained in a, that all the points in b are not
//                    found in a.  The striding of a may lead to it not
//                    referring to points in b.  Only do this calculation when
//                    absolutely necessary.
//
//-----------------------------------------------------------------------------

//
// The default (unit-stride) version of ContainsDomainSingle, which assumes
// that both arguments to 'contains' are 1D domains with unit stride
//

template<class T1, class T2, bool strided>
struct ContainsDomainSingle {
  static bool contains(const T1 &a, const T2 &b) {
    return (a.min() <= b.min() && a.max() >= b.max());
  }
};

//
// The non-unit-stride version of ContainsDomainSingle, which does extra
// work for the case where a and b do not have unit stride.
//

template<class T1, class T2>
struct ContainsDomainSingle<T1,T2,true> {
  static bool contains(const T1 &a, const T2 &b) {
    // types for elements in these domains
    typedef typename DomainTraits<T1>::Element_t E1_t;
    typedef typename DomainTraits<T2>::Element_t E2_t;

    // Find min and max values of type domains
    E1_t a0 = a.min();
    E1_t a1 = a.max();
    E1_t  s = a.stride();
    E2_t b0 = b.min();
    E2_t b1 = b.max();
    E2_t  t = b.stride();
    if (s < 0)
      s = -s;
    if (t < 0)
      t = -t;

    // We can do a quick short-circuit check to make sure they do not overlap
    // at all just from their endpoints.  If they don't even do this, we can
    // quit and say they do not touch.
    bool quicktest = (a0 <= b0 && a1 >= b1);
    if (!quicktest || s == 1)
      return quicktest;

    // OK, the endpoints of a contain those of b, and we must find out if
    // all the points in b are found in a.  This will be true if:
    //   1. The stride of b is a multiple of the stride of a
    //   2. The endpoints of b are found in a
    // If either of these conditions are false, a does not contain b 
    return (t % s == 0) && ((b0-a0) % s == 0) && ((a1-b1) % s == 0);
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of ContainsDomain:
//
// ContainsDomain implements a basic template meta-program to
// compare each dimension separately of the multidimensional domains for
// whether a contains b.  It uses ContainsDomainSingle to do the single-domain
// comparison, telling that struct whether the domains have unit stride
// or not.  A general version of ContainsDomain is defined, to compare the
// domains in the 'Dim' dimension, and then a specialization is provided
// for Dim==1 that stops the metaprogram recursion.
//
//-----------------------------------------------------------------------------

template<class T1, class T2, int Dim>
struct ContainsDomain {
  static bool contains(const T1 &a, const T2 &b) {
    // the types for the two domains used in ContainsDomainSingle may be a
    // little different than T1 and T2, since ContainsDomainSingle works
    // with 1D domains, not N-D domains.
    typedef typename DomainTraits<T1>::OneDomain_t Dom1_t;
    typedef typename DomainTraits<T2>::OneDomain_t Dom2_t;

    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T1>::unitStride;

    // which dimension we're working with
    const int n = Dim - 1;

    // compare the results for the 'Dim' dimension, and then for lower dims
    return ContainsDomainSingle<Dom1_t,Dom2_t,(USN == 0)>::contains(
      DomainTraits<T1>::getDomain(a,n), DomainTraits<T2>::getDomain(b,n)) &&
      ContainsDomain<T1,T2,n>::contains(a,b);
  }
};

template<class T1, class T2>
struct ContainsDomain<T1,T2,1> {
  static bool contains(const T1 &a, const T2 &b) {
    // the types for the two domains used in ContainsDomainSingle may be a
    // little different than T1 and T2, since ContainsDomainSingle works
    // with 1D domains, not N-D domains.
    typedef typename DomainTraits<T1>::OneDomain_t Dom1_t;
    typedef typename DomainTraits<T2>::OneDomain_t Dom2_t;

    // the number of relevant domains here which have unit stride
    const int USN = DomainTraits<T1>::unitStride;

    // compare the results for the 'Dim' dimension, and then for lower dims
    return ContainsDomainSingle<Dom1_t,Dom2_t,(USN == 0)>::contains(
      DomainTraits<T1>::getDomain(a,0), DomainTraits<T2>::getDomain(b,0));
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of contains:
//
// bool contains(domain1, domain2) is one of the domain calculus routines
// used to analyze domains to determine their relative characteristics.  It
// returns true if ALL the points in domain2 are found in the set of points
// which form domain1.  If so, this returns true, else it returns false.
//
// The implementation of contains is deferred to the ContainsDomain
// struct, which performs the contains comparison for each dimension and and's
// the results together.
//
//-----------------------------------------------------------------------------

template<class T1, class T2>
inline bool contains(const T1 &a, const T2 &b)
{
  CTAssert(int(DomainTraits<T1>::dimensions)
	   == int(DomainTraits<T2>::dimensions));
  return ContainsDomain<T1,T2,DomainTraits<T1>::dimensions>::contains(a, b);
}


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_CONTAINS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Contains.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:16:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
