// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Cartesian
//-----------------------------------------------------------------------------

#ifndef POOMA_COORDINATESYSYEMS_CARTESIAN_H
#define POOMA_COORDINATESYSYEMS_CARTESIAN_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Cartesian : An N-dimensional cartesian coordinate system.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Region.h"
#include "Tiny/Vector.h"

#include <iosfwd>

//-----------------------------------------------------------------------------
// Full Description:
// 
// Cartesian is an N-dimensional cartesian coordinate system, where the
// dimensionality is specified via the single template parameter.
//
// Exported typedefs:
//
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the coordinate system.
//
// Accessors:
//
//   distance(pt1, pt2) - returns the distance between two points.
//   print(stream) - prints information about this coordinate system to a
//     stream.
//   volume(Region) - returns the volume of the region.
//
//-----------------------------------------------------------------------------

template<int Dim>
class Cartesian
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef Cartesian<Dim> This_t;
 
  // The dimensionality of this coordinate system.
  
  enum { dimensions = Dim };

  //---------------------------------------------------------------------------
  // Volume calculation functions.

  // Compute the volume of a region, interpreted as a range of coordinate
  // values for each coordinate. In 1D, this is a length, in 2D an area, in 3D
  // a volume, in 4D a hypervolume, etc.

  template<class T>
  T volume(const Region<Dim, T> &r) const
  {
    return r.size();
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */
  
  //---------------------------------------------------------------------------
  // Distance calculation function.

  template<class T>
  T distance(const Vector<Dim, T> &pt1, const Vector<Dim, T> &pt2)
  {
    T dist = 0.0;
    for (int d = 0; d < Dim; d++) {
      dist += (pt2(d) - pt1(d))*(pt2(d) - pt1(d));
    }
    return sqrt(dist);
  }

  // --------------------------------------------------------------------------
  // Print a Cartesian<Dim> on an output stream.
  
  template <class Ostream>
  void print(Ostream &ostr) const
  {
    ostr << "Cartesian<" << Dim << ">";
  }

};

//-----------------------------------------------------------------------------
//
// ostream inserter for Cartesian<Dim>.
//
//-----------------------------------------------------------------------------

template<int Dim>
std::ostream &operator<<(std::ostream &ostr, 
  const Cartesian<Dim> &cs)
{
  cs.print(ostr);
  return ostr;
}

#endif // POOMA_COORDINATESYSYEMS_CARTESIAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Cartesian.h,v $   $Author: julianc $
// $Revision: 1.10 $   $Date: 2000/07/25 00:26:07 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
