// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_CONNECT_PAWS_RESIZE_H
#define POOMA_CONNECT_PAWS_RESIZE_H

//-----------------------------------------------------------------------------
// Classes:
// Resize<T>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// Resize<T> is a simple class with one static method 'resize' that is
// specialized on the type T to do an array resize operation for various
// engine types.  Some cannot be resized and will throw a runtime error,
// others will get their sizes changed (which may or may not preserve the
// existing contents).
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Utilities/PAssert.h"


///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
//
// Resize<T> : general version just throws a runtime assertion failure
//
//-----------------------------------------------------------------------------

template<class T>
struct Resize
{
  template<class Dom>
  static void resize(const T &, const Dom &)
  {
    PInsist(false, "Resize<T>::resize(): Cannot resize the given type.");
  }
};


//-----------------------------------------------------------------------------
//
// Resize<Array<T>> : For Array, by default try an initialize
//
//-----------------------------------------------------------------------------

template<int Dim, class T, class E>
struct Resize< Array<Dim, T, E> >
{
  template<class Dom>
  static void resize(Array<Dim, T, E> &array, const Dom &domain)
  {
    PInsist(false,
	    "Resize<Array<D,T,E>::resize(): Cannot resize the given type.");
  }
};


//-----------------------------------------------------------------------------
//
// Resize< Array<1,T,SharedBrick> : For special 1D engines,
// resize each patch separately to get an equal size in all patches.
//
//-----------------------------------------------------------------------------

template<class T>
struct Resize< Array<1, T, SharedBrick> >
{
  template<class Dom>
  static void resize(Array<1, T, SharedBrick> &array,
		     const Dom &domain)
  {
    // This will only work with 1D domains

    CTAssert(DomainTraits<Dom>::dimensions == 1);

    // Get the number of elements from the domain, and do resize

    int patchsize = DomainTraits<Dom>::getSize(domain);
    int currsize = array.domain().size();
    if (currsize > patchsize)
      {
	array.engine().destroy(Interval<1>(patchsize, currsize - 1), 0,
			       ShiftUp());
      }
    else if (currsize < patchsize)
      {
	array.engine().create(patchsize - currsize, 0);
      }

    // At the end, do a sync to get the layout updated.

    array.engine().sync();
  }
};


//-----------------------------------------------------------------------------
//
// Resize< Array<1,T,MultiBrick<Grid,Brick> > : For special 1D engines,
// resize each patch separately to get an equal size in all patches.
//
//-----------------------------------------------------------------------------

template<class T>
struct Resize< Array<1, T, MultiPatch<GridTag, Brick> > >
{
  template<class Dom>
  static void resize(Array<1, T, MultiPatch<GridTag, Brick> > &array,
		     const Dom &domain)
  {
    // This will only work with 1D domains

    CTAssert(DomainTraits<Dom>::dimensions == 1);

    // We must have some patches in the destination array

    int patches = array.numPatchesLocal();
    PAssert(patches > 0);

    // Get the number of elements from the domain

    int newsize = DomainTraits<Dom>::getSize(domain);

    // Compute the number of elements to put in each patch

    int basicpatchsize = newsize / patches;
    int extra = newsize % patches;

    // Resize each patch ... this could be made parallel since each resize
    // is independent.

    for (int p=0; p < patches; ++p)
      {
	int patchsize = basicpatchsize + (p < extra ? 1 : 0);
	int currsize = array.patch(p).domain().size();
	if (currsize > patchsize)
	  {
	    array.engine().destroy(Interval<1>(patchsize, currsize - 1), p,
				   ShiftUp());
	  }
	else if (currsize < patchsize)
	  {
	    array.engine().create(patchsize - currsize, p);
	  }
      }

    // At the end, do a sync to get the layout updated.

    array.engine().sync();
  }
};


// }; // namespace Pooma
///////////////////////////////////////////////////////////////////////////////


#endif // POOMA_CONNECT_PAWS_RESIZE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Resize.h,v $   $Author: julianc $
// $Revision: 1.4 $   $Date: 2000/06/20 17:09:38 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
