#include "Pooma/Arrays.h"

#include "BConds/BCond.h"
#include "BConds/BCondBase.h"
#include "BConds/BCondList.h"

//notyet include "Engine/CoordinateLatticeEngine.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Here is an example of a boundary condition category class.
//-----------------------------------------------------------------------------

template <class T>
class ExtrapolateFaceBC : public BCondCategory<ExtrapolateFaceBC<T> >
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ExtrapolateFaceBC(int face, T offset, T slope) : 
    face_m(face), offset_m(offset), slope_m(slope) { }

  ExtrapolateFaceBC(const ExtrapolateFaceBC &model) : 
    face_m(model.face()), offset_m(model.offset()), slope_m(model.slope()) { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ExtrapolateFaceBC &operator=(const ExtrapolateFaceBC &rhs)
  {
    face_m = rhs.face();
    offset_m = rhs.offset();
    slope_m = rhs.slope();
    
    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }
  T offset() const { return offset_m; }
  T slope() const { return slope_m; }
  
private:

  int face_m;
  T offset_m;
  T slope_m;
};

//-----------------------------------------------------------------------------
// Here is an example of a boundary condition class.
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag, class T1>
class BCond<Array<Dim, T, EngineTag>, ExtrapolateFaceBC<T1> > :
  public BCondBase<Interval<Dim>, Array<Dim, T, EngineTag> >
{
public:

  typedef BCondBase<Interval<Dim>, Array<Dim, T, EngineTag> > Base_t;
  typedef typename Base_t::Domain_t Domain_t;
  
  BCond(const Array<Dim, T, EngineTag> &a, const ExtrapolateFaceBC<T1> &bc)
    : Base_t(a, a.domain(), a.domain()), bc_m(&bc)
  {
    Domain_t &dest = destDomain();
    Domain_t &src = srcDomain();
    
    int nGuards; // Number of global guard layers on the face in question.

    int d = bc.face() / 2;
    if (bc.face() & 1)
      {
        // High face.

	//stub	nGuards = rightGlobalGuardElements(F, direction);
	nGuards = 1; // Hardwire one guard layer, for now.
	
	// Using Intervals here assumes the Field's domain has unit stride, no?

	// Specialize for cell-centering, for now: The "+/- one" business is
	// accounting for using Arrays with no global guards; it goes away
	// when we have global guards.
	int one = 1;
        
        dest[d] = Interval<1>(dest[d].max() + 1 - one, 
			      dest[d].max() + nGuards - one);
        src[d] = Interval<1>(src[d].max() - one - (nGuards - 1), 
			     src[d].max() - one);
      }
    else
      {
        // Low face.
        
	//stub	nGuards = rightGlobalGuardElements(F, direction);
	nGuards = 1; // Hardwire one guard layer, for now.
	
	// Using Intervals here assumes the Field's domain has unit stride, no?

	// Specialize for cell-centering, for now: The "+/- one" business is
	// accounting for using Arrays with no global guards; it goes away
	// when we have global guards.
	int one = 1;

        dest[d] = Interval<1>(dest[d].min() + one - nGuards, 
			      dest[d].min() + one - 1);

        src[d] = Interval<1>(src[d].min() + one, 
			     src[d].min() + one + (nGuards - 1));
      }
  }
  
  void applyBoundaryCondition()
  {
    std::cout << "destDomain() = " << destDomain()
	      << " ; srcDomain() = " << srcDomain()
	      << " ; bc_m->slope() = " << bc_m->slope()
	      << " ; bc_m->offset() = " << bc_m->offset() << std::endl;
    subject()(destDomain()) = 
      subject()(srcDomain()) * bc_m->slope() + bc_m->offset();
  }

private:

  const ExtrapolateFaceBC<T1> *bc_m;
};


int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  
  Array<2> a(6, 6);
  //   a = 3.0;
  //notyet  Interval<1> I(6);
  //notyet  Interval<1> J(6);
  //notyet  a = I + J;
  for (int i=0; i<6; i++) {
    for (int j=0; j<6; j++) {
      a(i,j) = i + j;
    }
  }
  std::cout << a << std::endl;

  // NegReflectFace, scalar (double) offset and slope:
  ExtrapolateFaceBC<double> zf(1, 0.0, -1.0);
  
  BCondItem *bcScalar = zf.create(a);  
  
  Range<1> r(0,4), s(0,5);
  Range<2> rs(r, s), ss(s, s);
  
  bcScalar->notifyPreRead(rs);
  std::cout << a << std::endl;
  
  bcScalar->notifyPreRead(ss);
  std::cout << a << std::endl;
  
  Array<2, Vector<3> > b(6, 6);
  //  b = Vector<3>(1.0, 2.0, 3.0);
  //notyet  Array<2,Loc<2>,CoordinateLattice> coordinates;
  //notyet  b = coordinates.comp(0) + coordinates.comp(1);
  for (int i=0; i<6; i++) {
    for (int j=0; j<6; j++) {
      for (int c=0; c<3; c++) {
	(b.comp(c))(i,j) = i + j + c;
      }
    }
  }
  std::cout << b << std::endl;
  
  // VectorFace, componentwise ==> double offset and slope:

  BCondList vbc;

  // Low X face:

  // Make negative and positive reflecting BC types:
  ExtrapolateFaceBC<double> nrf0(0, 0.0, -1.0);
  ExtrapolateFaceBC<double> prf0(0, 0.0, +1.0);

  // Use these componentwise:
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(0, nrf0));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(1, prf0));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(2, prf0));

  // High X face:

  // Make negative and positive reflecting BC types:
  ExtrapolateFaceBC<double> nrf1(1, 0.0, -1.0);
  ExtrapolateFaceBC<double> prf1(1, 0.0, +1.0);

  // Use these componentwise:
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(0, nrf1));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(1, prf1));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(2, prf1));

  // Low Y face:

  // Make negative and positive reflecting BC types:
  ExtrapolateFaceBC<double> nrf2(2, 0.0, -1.0);
  ExtrapolateFaceBC<double> prf2(2, 0.0, +1.0);

  // Use these componentwise:
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(0, prf2));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(1, nrf2));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(2, prf2));

  // High Y face:

  // Make negative and positive reflecting BC types:
  ExtrapolateFaceBC<double> nrf3(3, 0.0, -1.0);
  ExtrapolateFaceBC<double> prf3(3, 0.0, +1.0);

  // Use these componentwise:
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(0, prf3));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(1, nrf3));
  vbc.addBoundaryCondition(b,
    ComponentBC< 1, ExtrapolateFaceBC<double> >(2, prf3));
  
  vbc.notifyPreRead(rs);
  std::cout << b << std::endl;
  
  vbc.notifyPreRead(ss);
  std::cout << b << std::endl;
  
  Pooma::finalize();

  return 0;
}

// $Id: bc_test2.cpp,v 1.7 1999/09/23 18:43:12 julianc Exp $
