# \file cmake/semver-functions.cmake

# semver_set_buildmetadata(<var> "+build.metadata")
# => set(<var> "build" "metadata")
function (semver_set_buildmetadata var str)
    set(buildmetadata)
    if (str)
        if (NOT "${str}" MATCHES "^\\+.")
            message(FATAL_ERROR "semver_set_buildmetadata: invalid buildmetadata '${str}'")
        endif()
        string(REGEX REPLACE "^\\+(.*)$" "\\1" buildmetadata "${str}")
        string(REPLACE "." ";" buildmetadata "${buildmetadata}")
        foreach (value IN LISTS buildmetadata)
            if (NOT "${value}" MATCHES "^[a-zA-Z0-9\\-]+$")
                message(FATAL_ERROR "semver_set_buildmetadata: invalid buildmetadata value '${value}' in '${str}'")
            endif()
        endforeach()
    endif()
    set("${var}" "${buildmetadata}" PARENT_SCOPE)
endfunction()

# semver_set(<var> "-pre.release")
# => set(<var> "pre" "release")
function (semver_set_prerelease var str)
    set(prerelease)
    if (str)
        if (NOT "${str}" MATCHES "^\\-.")
            message(FATAL_ERROR "semver_set_prerelease: invalid prerelease '${str}'")
        endif()
        string(REGEX REPLACE "^\\-(.*)$" "\\1" prerelease "${str}")
        string(REPLACE "." ";" prerelease "${prerelease}")
        foreach (value IN LISTS prerelease)
            if (NOT "${value}" MATCHES "^[a-zA-Z0-9]+$")
                message(FATAL_ERROR "semver_set_prerelease: invalid prerelease value '${value}' in '${str}'")
            endif()
        endforeach()
    endif()
    set("${var}" "${prerelease}" PARENT_SCOPE)
endfunction()

# semver_set(<var> "0.1.2-pre.release+build.metadata")
# => set(<var> "0.1.2-pre.release+build.metadata")
# => set(<var>_MAJOR "0")
# => set(<var>_MINOR "1")
# => set(<var>_MINOR "2")
# => set(<var>_PRERELEASE "pre" "release")
# => set(<var>_BUILDMETADATA "build" "metadata")
function (semver_set var str)
    set(tmp "${str}")
    set(buildmetadata)
    if ("${tmp}" MATCHES "\\+")
        string(REGEX REPLACE "^[^\\+]*(\\+.*)$" "\\1" buildmetadata "${tmp}")
        semver_set_buildmetadata(buildmetadata "${buildmetadata}")
        string(REGEX REPLACE "\\+.*$" "" tmp "${tmp}")
    endif()
    set(prerelease)
    if ("${tmp}" MATCHES "\\-")
        string(REGEX REPLACE "^[^\\-]*(\\-.*)$" "\\1" prerelease "${tmp}")
        semver_set_prerelease(prerelease "${prerelease}")
        string(REGEX REPLACE "\\-.*$" "" tmp "${tmp}")
    endif()
    if (NOT "${tmp}" MATCHES "^[0-9]+\\.[0-9]+\\.[0-9]+$")
        message(FATAL_ERROR "semver_set: invalid version '${tmp}'")
    endif()
    string(REGEX REPLACE "^[0-9]+\\.[0-9]+\\.([0-9]+)$" "\\1" patch "${tmp}")
    string(REGEX REPLACE "^[0-9]+\\.([0-9]+)\\..*$" "\\1" minor "${tmp}")
    string(REGEX REPLACE "^([0-9]+)\\..*$" "\\1" major "${tmp}")
    set("${var}" "${str}" PARENT_SCOPE)
    set("${var}_MAJOR" "${major}" PARENT_SCOPE)
    set("${var}_MINOR" "${minor}" PARENT_SCOPE)
    set("${var}_PATCH" "${patch}" PARENT_SCOPE)
    set("${var}_PRERELEASE" "${prerelease}" PARENT_SCOPE)
    set("${var}_BUILDMETADATA" "${buildmetadata}" PARENT_SCOPE)
endfunction()

# semver_rebuild(<var>)
# => set(<var> "...")
function (semver_rebuild var)
    set(tmp "${${var}_MAJOR}.${${var}_MINOR}.${${var}_PATCH}")
    if (NOT "${tmp}" MATCHES "^[0-9]+\\.[0-9]+\\.[0-9]+$")
        message(FATAL_ERROR "semver_rebuild: invalid version '${tmp}'")
    endif()
    set(prerelease "${${var}_PRERELEASE}")
    if (prerelease)
        foreach (value IN LISTS prerelease)
            if (NOT "${value}" MATCHES "^[a-zA-Z0-9]+$")
                message(FATAL_ERROR "semver_rebuild: invalid prerelease '${prerelease}'")
            endif()
        endforeach()
        string(REPLACE ";" "." prerelease "${prerelease}")
        set(tmp "${tmp}-${prerelease}")
    endif()
    set(buildmetadata "${${var}_BUILDMETADATA}")
    if (buildmetadata)
        foreach (value IN LISTS buildmetadata)
            if (NOT "${value}" MATCHES "^[a-zA-Z0-9\\-]+$")
                message(FATAL_ERROR "semver_rebuild: invalid buildmetadata '${buildmetadata}'")
            endif()
        endforeach()
        string(REPLACE ";" "." buildmetadata "${buildmetadata}")
        set(tmp "${tmp}+${buildmetadata}")
    endif()
    set("${var}" "${tmp}" PARENT_SCOPE)
endfunction()
