/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.java;

import org.apiguardian.api.API;
import org.opentest4j.reporting.events.api.Appendable;
import org.opentest4j.reporting.events.api.ChildElement;
import org.opentest4j.reporting.events.api.Context;
import org.opentest4j.reporting.events.core.CoreFactory;
import org.opentest4j.reporting.events.core.Sources;
import org.opentest4j.reporting.schema.Namespace;
import org.opentest4j.reporting.schema.QualifiedName;

import java.util.Optional;

import static org.apiguardian.api.API.Status.EXPERIMENTAL;

/**
 * The {@code classpathResourceSource} element of the Java namespace.
 *
 * @since 0.1.0
 */
@API(status = EXPERIMENTAL, since = "0.1.0")
public class ClasspathResourceSource extends ChildElement<Sources, ClasspathResourceSource> {

	private static final QualifiedName ELEMENT = QualifiedName.of(Namespace.REPORTING_JAVA, "classpathResourceSource");
	private static final QualifiedName RESOURCE_NAME = QualifiedName.of(Namespace.REPORTING_JAVA, "resourceName");

	ClasspathResourceSource(Context context) {
		super(context, ClasspathResourceSource.ELEMENT);
	}

	ClasspathResourceSource withResourceName(String resourceName) {
		withAttribute(RESOURCE_NAME, resourceName);
		return this;
	}

	/**
	 * Add a {@code filePosition} child element to this element.
	 *
	 * @param line   the line number
	 * @param column the column number
	 * @return this element
	 */
	public Appendable<ClasspathResourceSource> addFilePosition(int line, Optional<Integer> column) {
		return append(CoreFactory.filePosition(line, column));
	}
}
