/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.viewController;

/**
 * An interface for use with the InterfaceViewControllerExecutor.
 * <p>
 * When the InterfaceViewControllerExecutor has been selected as the technique for mapping
 * lifecycle events onto bean methods, then the target bean must implement this interface
 * in order to get callbacks.
 * <p>
 * Note that the expression "view controller bean" is used to generally refer to an object
 * that can handle lifecycle events for a view. Such beans may implement this interface, but
 * may not - they might be accessed via the ReflectiveViewControllerExecutor or other
 * implementations that do not require this interface. 
 */
public interface ViewController
{
    /**
     * This method will <i>always</i> be called before any other method is invoked
     * on any backing bean for the current request. It is invoked once for each
     * request.
     * <p>
     * This callback could possibly be better named "notifyBeginRequestUsingThisView"
     * or similar, as it is invoked per request.
     * <p>
     * There are three different situations in which initView callbacks occur:
     * <ol>
     * <li>
     * A view is just being rendered.<br>
     * The initView callback gets called once (at BeforeRender)
     * </li>
     * <li>
     * A postback is processed, no navigation occurs.<br>
     * The initView callback gets called once (at AfterRestoreView)
     * </li>
     * <li>
     * A postback occurs, navigation to a different view instance occurs.<br>
     * The initView callback gets called once for the original view, and then once for the new view.
     * </li>
     * </ol>
     * <p>
     * Note that the condition tested is whether the <i>view instance</i> has changed; if navigation causes
     * a new view root to be created then the initView callback occurs even if that view root has the
     * same viewId [1].
     * <p>
     * Note [1]: Orchestra versions 1.3 and earlier test only the viewId string.
     */
    public void initView();

    /**
     * This method will be invoked before any "action" callbacks related to buttons,
     * links, etc. are invoked on backing beans for the current request.
     * <p>
     * For JSF, there is an exception: for command components marked as "immediate",
     * the associated action method is invoked before this callback occurs. If that
     * method then performs navigation then this callback will not occur at all.
     * <p>
     * Note also that for JSF, if validation failures occur then this callback will
     * not be invoked (as actions are skipped).  
     */
    public void preProcess();
    
    /**
     * This method will be invoked just before starting to render output to the user.
     */
    public void preRenderView();
}
