/*
 * Copyright 2012 Splunk, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"): you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package com.splunk;

/**
 * The {@code JobResultsArgs} class contains arguments for getting job results
 * using the {@link Job#getResults} method.
 */
public class JobResultsArgs extends Args {

    /**
     * Specifies the format for the returned output.
     */
    public static enum OutputMode {
        /** Returns output in Atom format. */
        ATOM("atom"),
        /** Returns output in CSV format. */
        CSV("csv"),
        /** Returns output in JSON format. */
        JSON("json"),
        /** Returns output in JSON_COLS format. */
        JSON_COLS("json_cols"),
        /** Returns output in JSON_ROWS format. */
        JSON_ROWS("json_rows"),
        /** Returns output in raw format. */
        RAW("raw"),
        /** Returns output in XML format. */
        XML("xml");
        
        private String value;
        
        private OutputMode(String value) {
            this.value = value;
        }
        
        /**
         * @return The REST API value for this enumerated constant.
         */
        public String toString() {
            return this.value;
        }
    }

    /**
     * Class constructor.
     */
    public JobResultsArgs() { super(); }
    
    /* BEGIN AUTOGENERATED CODE */
    
    /**
     * Sets the maximum number of results to return. 
     * 
     * @param count
     *      The maximum number of results. To return all available results, specify 0.
     */
    public void setCount(int count) {
        this.put("count", count);
    }
    
    /**
     * Sets a list of fields to return for the event set. 
     * 
     * @param fieldList
     *      A list of fields.
     */
    public void setFieldList(String[] fieldList) {
        this.put("f", fieldList);
    }
    
    /**
     * Specifies the index of the first result (inclusive) from which to begin returning data. This value is 0-indexed.<p>In Splunk 4.1+, negative offsets are allowed and are added to the count to compute the absolute offset (for example, offset=-1 is the last available offset). Offsets in the results are always absolute and never negative. The default value is 0.
     * 
     * @param offset
     *      The index of the first result to return.
     */
    public void setOffset(int offset) {
        this.put("offset", offset);
    }
    
    /**
     * Sets the format of the output.
     * 
     * @param outputMode
     *      The output format.
     */
    public void setOutputMode(OutputMode outputMode) {
        this.put("output_mode", outputMode);
    }
    
    /**
     * Sets the post-processing search to apply to results. 
     * 
     * @param search
     *      The post-processing search query.
     */
    public void setSearch(String search) {
        this.put("search", search);
    }
    
    /* END AUTOGENERATED CODE */
}
