package org.apache.torque.adapter;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/**
 * Interface for various ID retrieval method types
 * (i.e. auto-increment, sequence, ID broker, etc.).
 *
 * @author <a href="mailto:dlr@collab.net">Daniel Rall</a>
 * @version $Id: IDMethod.java 1336543 2012-05-10 08:56:52Z tfischer $
 */
public enum IDMethod
{
    /**
     * Key generation via database-specific ID method
     * (i.e. auto-increment for MySQL, sequence for Oracle, etc.).
     */
    NATIVE("native"),

    /**
     * Key generation via auto-increment.
     */
    AUTO_INCREMENT("autoincrement"),

    /**
     * Key generation via sequences.
     */
    SEQUENCE("sequence"),

    /**
     * Key generation via the IDBroker table.
     */
    ID_BROKER("idbroker"),

    /**
     * No RDBMS key generation (keys may be generated by the application).
     */
    NO_ID_METHOD("none");

    /** The name of the id method. */
    private String idMethod;

    /**
     * Constructor.
     *
     * @param idMethod the text of the idMethod.
     */
    private IDMethod(String idMethod)
    {
        this.idMethod = idMethod;
    }

    @Override
    public String toString()
    {
        return idMethod;
    }

    /**
     * Returns the id method for a id method name.
     *
     * @param idMethod the name of the id method.
     *
     * @return the id method.
     *
     * @throws IllegalArgumentException if idMethod is not a known name.
     */
    public static IDMethod getIdMethod(String idMethod)
    {
        for (IDMethod candidate : values())
        {
            if (candidate.toString().equals(idMethod))
            {
                return candidate;
            }
        }
        throw new IllegalArgumentException("Unknown idMethod " + idMethod);
    }
}
